FreeSpoolWinch = {}
local FreeSpoolWinch_mt = Class(FreeSpoolWinch)

-- Require base Winch specialization
function FreeSpoolWinch.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(Winch, specializations)
end

function FreeSpoolWinch.initSpecialization()
    local schema = Vehicle.xmlSchema
    schema:setXMLSpecializationType("FreeSpoolWinch")

    -- Free spool tuning
    schema:register(XMLValueType.FLOAT,
        "vehicle.freeSpoolWinch#freeSpoolSpeedFactor",
        "Free spool speed factor",
        1.0
    )

    schema:register(XMLValueType.FLOAT,
        "vehicle.freeSpoolWinch#slackDistance",
        "Slack distance before free spool (smaller = more sensitive)",
        0.05
    )

    -- Custom HUD/interaction text overrides
    -- Values can be:
    --  - a direct string: "Pull rigging line"
    --  - or a l10n key present in modDesc: "myRig_pull"
    schema:register(XMLValueType.STRING,
        "vehicle.freeSpoolWinch#pullRopeText",
        "Custom text or l10n key for 'pull rope' control"
    )
    schema:register(XMLValueType.STRING,
        "vehicle.freeSpoolWinch#attachRopeText",
        "Custom text or l10n key for 'attach rope'"
    )
    schema:register(XMLValueType.STRING,
        "vehicle.freeSpoolWinch#attachAnotherRopeText",
        "Custom text or l10n key for 'attach another rope'"
    )
    schema:register(XMLValueType.STRING,
        "vehicle.freeSpoolWinch#detachRopeText",
        "Custom text or l10n key for 'detach rope'"
    )

    schema:setXMLSpecializationType()
end

function FreeSpoolWinch.registerFunctions(vehicleType)
    SpecializationUtil.registerFunction(vehicleType, "setFreeSpoolEnabled", FreeSpoolWinch.setFreeSpoolEnabled)
    SpecializationUtil.registerFunction(vehicleType, "getFreeSpoolEnabled", FreeSpoolWinch.getFreeSpoolEnabled)
end

function FreeSpoolWinch.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onLoad", FreeSpoolWinch)
    SpecializationUtil.registerEventListener(vehicleType, "onRegisterActionEvents", FreeSpoolWinch)
    SpecializationUtil.registerEventListener(vehicleType, "onPostUpdate", FreeSpoolWinch)
end

-- Helper: resolve either a raw string or treat as i18n key
local function resolveText(value)
    if value == nil or value == "" then
        return nil
    end
    if g_i18n ~= nil and g_i18n:hasText(value) then
        return g_i18n:getText(value)
    end
    return value
end

-- LOAD -----------------------------------------------------------------------

function FreeSpoolWinch:onLoad(savegame)
    local spec = {}
    self.spec_freeSpoolWinch = spec

    spec.enabled = false
    spec.lastToggleTime = 0
    spec.toggleDelay = 250 -- ms debounce

    spec.speedFactor   = self.xmlFile:getValue("vehicle.freeSpoolWinch#freeSpoolSpeedFactor", 1.0)
    spec.slackDistance = self.xmlFile:getValue("vehicle.freeSpoolWinch#slackDistance", 0.05)

    spec.actionEvents = {}
    spec.freeSpool = {} -- [ropeIndex] = { allowedLength = number }

    -- Read custom text config
    spec.customPullText          = self.xmlFile:getValue("vehicle.freeSpoolWinch#pullRopeText")
    spec.customAttachText        = self.xmlFile:getValue("vehicle.freeSpoolWinch#attachRopeText")
    spec.customAttachAnotherText = self.xmlFile:getValue("vehicle.freeSpoolWinch#attachAnotherRopeText")
    spec.customDetachText        = self.xmlFile:getValue("vehicle.freeSpoolWinch#detachRopeText")

    -- Apply text overrides onto Winch spec texts (if present)
    local winchSpec = self.spec_winch
    if winchSpec ~= nil and winchSpec.texts ~= nil then
        local texts = winchSpec.texts

        local pullText = resolveText(spec.customPullText)
        if pullText ~= nil then
            texts.control = pullText
        end

        local attachText = resolveText(spec.customAttachText)
        if attachText ~= nil then
            texts.attachTree = attachText
            -- if no explicit "attach another" was set, mirror this
            if spec.customAttachAnotherText == nil then
                texts.attachAnotherTree = attachText
            end
        end

        local attachAnotherText = resolveText(spec.customAttachAnotherText)
        if attachAnotherText ~= nil then
            texts.attachAnotherTree = attachAnotherText
        end

        local detachText = resolveText(spec.customDetachText)
        if detachText ~= nil then
            texts.detachTree = detachText
        end
    end
end

-- INPUT / UI -----------------------------------------------------------------

function FreeSpoolWinch:onRegisterActionEvents(isActiveForInput, isActiveForInputIgnoreSelection)
    if not self.isClient then
        return
    end

    local winchSpec = self.spec_winch
    local spec = self.spec_freeSpoolWinch
    if spec == nil or winchSpec == nil or winchSpec.ropes == nil or #winchSpec.ropes == 0 then
        return
    end

    self:clearActionEventsTable(spec.actionEvents)

    if isActiveForInputIgnoreSelection then
        local _, actionEventId = self:addActionEvent(
            spec.actionEvents,
            InputAction.WINCH_FREESPOOL,
            self,
            FreeSpoolWinch.actionEventToggleFreeSpool,
            false, true, false, true, nil
        )

        if actionEventId ~= nil then
            g_inputBinding:setActionEventTextPriority(actionEventId, GS_PRIO_VERY_HIGH)
            FreeSpoolWinch.updateActionEventText(self)
        end
    end
end

function FreeSpoolWinch.updateActionEventText(self)
    local spec = self.spec_freeSpoolWinch
    if not self.isClient or spec == nil or spec.actionEvents == nil then
        return
    end

    local ae = spec.actionEvents[InputAction.WINCH_FREESPOOL]
    if ae ~= nil then
        local text
        if spec.enabled then
            text = g_i18n:hasText("action_winch_freeSpool_off")
                and g_i18n:getText("action_winch_freeSpool_off")
                or "Winch Free-Spool Mode Disable"
        else
            text = g_i18n:hasText("action_winch_freeSpool_on")
                and g_i18n:getText("action_winch_freeSpool_on")
                or "Winch Free-Spool Mode Enable"
        end
        g_inputBinding:setActionEventText(ae.actionEventId, text)
    end
end

function FreeSpoolWinch.actionEventToggleFreeSpool(self, actionName, inputValue, callbackState, isAnalog)
    local spec = self.spec_freeSpoolWinch
    if spec == nil then
        return
    end

    local now = g_time or 0
    if now - spec.lastToggleTime < spec.toggleDelay then
        return
    end
    spec.lastToggleTime = now

    self:setFreeSpoolEnabled(not spec.enabled)
end

-- API ------------------------------------------------------------------------

function FreeSpoolWinch:setFreeSpoolEnabled(state)
    local spec = self.spec_freeSpoolWinch
    if spec == nil then
        return
    end

    state = not not state
    if spec.enabled == state then
        return
    end

    spec.enabled = state

    if not spec.enabled then
        spec.freeSpool = {}
    end

    if self.isClient then
        FreeSpoolWinch.updateActionEventText(self)

        -- Support multiple possible key variants
        local keyEnabled =
            g_i18n:hasText("info_winch_freeSpool_enabled") and "info_winch_freeSpool_enabled"
            or g_i18n:hasText("info_winch_freespool_enable") and "info_winch_freespool_enable"
            or nil

        local keyDisabled =
            g_i18n:hasText("info_winch_freeSpool_disabled") and "info_winch_freeSpool_disabled"
            or g_i18n:hasText("info_winch_freespool_disabled") and "info_winch_freespool_disabled"
            or nil

        local msg
        if spec.enabled then
            msg = keyEnabled and g_i18n:getText(keyEnabled) or "Winch Free-Spool Enabled"
        else
            msg = keyDisabled and g_i18n:getText(keyDisabled) or "Winch Free-Spool Disabled"
        end

        g_currentMission:showBlinkingWarning(msg, 1500)
    end
end

function FreeSpoolWinch:getFreeSpoolEnabled()
    local spec = self.spec_freeSpoolWinch
    return spec ~= nil and spec.enabled
end

-- ATTACH TARGET RESOLUTION (supports vanilla + generic attach mods) ----------

local function getAttachmentTarget(rope)
    -- Vanilla / BetterWinch-style trees
    if rope.attachedTrees ~= nil and #rope.attachedTrees > 0 then
        local a = rope.attachedTrees[1]
        if a.activeHookData ~= nil then
            if a.activeHookData.getRopeTargetPosition ~= nil then
                local x, y, z = a.activeHookData:getRopeTargetPosition()
                return true, x, y, z
            end
            if a.activeHookData.hookId ~= nil and entityExists(a.activeHookData.hookId) then
                local x, y, z = getWorldTranslation(a.activeHookData.hookId)
                return true, x, y, z
            end
        end
    end

    -- Generic attached objects
    if rope.attachedObjects ~= nil and #rope.attachedObjects > 0 then
        local o = rope.attachedObjects[1]
        if o.getRopeTargetPosition ~= nil then
            local x, y, z = o:getRopeTargetPosition()
            return true, x, y, z
        end
        if o.node ~= nil and entityExists(o.node) then
            local x, y, z = getWorldTranslation(o.node)
            return true, x, y, z
        end
    end

    -- Generic attached vehicle
    if rope.attachedVehicle ~= nil then
        local v = rope.attachedVehicle
        if v.getRopeTargetPosition ~= nil then
            local x, y, z = v:getRopeTargetPosition()
            return true, x, y, z
        end
        if v.rootNode ~= nil and entityExists(v.rootNode) then
            local x, y, z = getWorldTranslation(v.rootNode)
            return true, x, y, z
        end
    end

    return false, 0, 0, 0
end

-- RUNTIME: tension-based free spool (server-side) ----------------------------

function FreeSpoolWinch:onPostUpdate(dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
    local spec = self.spec_freeSpoolWinch
    local winchSpec = self.spec_winch

    if spec == nil or winchSpec == nil or not spec.enabled or not self.isServer then
        return
    end

    local slack       = spec.slackDistance or 0.05
    local speedFactor = spec.speedFactor or 1.0

    for i, rope in ipairs(winchSpec.ropes) do
        if rope.mainRope == nil or rope.ropeNode == nil then
            spec.freeSpool[i] = nil
        else
            local hasTarget, hx, hy, hz = getAttachmentTarget(rope)
            if not hasTarget then
                spec.freeSpool[i] = nil
            else
                local rx, ry, rz = getWorldTranslation(rope.ropeNode)
                local dist = MathUtil.vector3Length(hx - rx, hy - ry, hz - rz)

                local fs = spec.freeSpool[i]
                if fs == nil then
                    fs = { allowedLength = dist }
                    spec.freeSpool[i] = fs
                end

                local allowed = fs.allowedLength or dist

                -- Relax baseline slightly when slack so new small pulls register
                if dist < allowed then
                    local relaxRate = (rope.speed or 0.001) * 0.5
                    local relax = relaxRate * dt
                    if relax > 0 then
                        allowed = math.max(dist, allowed - relax)
                        fs.allowedLength = allowed
                    end
                end

                -- If trying to go longer than allowed + slack => feed rope out
                if dist > allowed + slack then
                    local ropeSpeed = rope.speed or 0.001
                    local delta = ropeSpeed * speedFactor * dt

                    if delta > 0 then
                        local maxLen = rope.maxLength or math.huge
                        if allowed + delta > maxLen then
                            delta = math.max(0, maxLen - allowed)
                        end

                        if delta > 0 then
                            local changed = rope.mainRope:adjustLength(delta)
                            if changed ~= 0 then
                                fs.allowedLength = allowed + changed

                                rope.controlDirection = -1
                                rope.lastControlTimer = 100

                                if winchSpec.dirtyFlag ~= nil then
                                    self:raiseDirtyFlags(winchSpec.dirtyFlag)
                                end
                                if winchSpec.ropeDirtyFlag ~= nil then
                                    self:raiseDirtyFlags(winchSpec.ropeDirtyFlag)
                                end
                            end
                        end
                    end
                else
                    -- No significant tension: don't force pull-in, just stop release drive
                    if rope.controlDirection < 0 and rope.lastControlTimer <= 0 then
                        rope.controlDirection = 0
                    end
                end
            end
        end
    end
end
