FreeSpoolWinch = {}
local FreeSpoolWinch_mt = Class(FreeSpoolWinch)

-- Require base Winch spec
function FreeSpoolWinch.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(Winch, specializations)
end

function FreeSpoolWinch.initSpecialization()
    local schema = Vehicle.xmlSchema
    schema:setXMLSpecializationType("FreeSpoolWinch")

    schema:register(XMLValueType.FLOAT,
        "vehicle.freeSpoolWinch#freeSpoolSpeedFactor",
        "Free spool speed factor",
        1.0
    )

    schema:register(XMLValueType.FLOAT,
        "vehicle.freeSpoolWinch#slackDistance",
        "Slack distance in meters before free spool extends",
        0.15
    )

    schema:setXMLSpecializationType()
end

function FreeSpoolWinch.registerFunctions(vehicleType)
    SpecializationUtil.registerFunction(vehicleType, "setFreeSpoolEnabled", FreeSpoolWinch.setFreeSpoolEnabled)
    SpecializationUtil.registerFunction(vehicleType, "getFreeSpoolEnabled", FreeSpoolWinch.getFreeSpoolEnabled)
end

function FreeSpoolWinch.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onLoad", FreeSpoolWinch)
    SpecializationUtil.registerEventListener(vehicleType, "onRegisterActionEvents", FreeSpoolWinch)
    SpecializationUtil.registerEventListener(vehicleType, "onPostUpdate", FreeSpoolWinch)
end

-- LOAD -----------------------------------------------------------------------

function FreeSpoolWinch:onLoad(savegame)
    local spec = {}
    self.spec_freeSpoolWinch = spec

    spec.enabled = false
    spec.lastToggleTime = 0
    spec.toggleDelay = 250 -- ms

    spec.speedFactor = self.xmlFile:getValue("vehicle.freeSpoolWinch#freeSpoolSpeedFactor", 1.0)
    spec.slackDistance = self.xmlFile:getValue("vehicle.freeSpoolWinch#slackDistance", 0.15)

    spec.actionEvents = {}
    spec.freeSpool = {} -- [ropeIndex] = { allowedLength = number }
end

-- INPUT / UI -----------------------------------------------------------------

function FreeSpoolWinch:onRegisterActionEvents(isActiveForInput, isActiveForInputIgnoreSelection)
    if not self.isClient then
        return
    end

    local winchSpec = self.spec_winch
    local spec = self.spec_freeSpoolWinch
    if winchSpec == nil or winchSpec.ropes == nil or #winchSpec.ropes == 0 then
        return
    end

    self:clearActionEventsTable(spec.actionEvents)

    if isActiveForInputIgnoreSelection then
        local _, actionEventId = self:addActionEvent(
            spec.actionEvents,
            InputAction.WINCH_FREESPOOL,
            self,
            FreeSpoolWinch.actionEventToggleFreeSpool,
            false, true, false, true, nil
        )

        if actionEventId ~= nil then
            g_inputBinding:setActionEventTextPriority(actionEventId, GS_PRIO_VERY_HIGH)
            FreeSpoolWinch.updateActionEventText(self)
        end
    end
end

function FreeSpoolWinch.updateActionEventText(self)
    local spec = self.spec_freeSpoolWinch
    if not self.isClient or spec.actionEvents == nil then
        return
    end

    local ae = spec.actionEvents[InputAction.WINCH_FREESPOOL]
    if ae ~= nil then
        local text
        if spec.enabled then
            text = g_i18n:hasText("action_winch_freeSpool_off")
                and g_i18n:getText("action_winch_freeSpool_off")
                or "Winch Free-Spool Mode Disable"
        else
            text = g_i18n:hasText("action_winch_freeSpool_on")
                and g_i18n:getText("action_winch_freeSpool_on")
                or "Winch Free-Spool Mode Enable"
        end
        g_inputBinding:setActionEventText(ae.actionEventId, text)
    end
end

function FreeSpoolWinch.actionEventToggleFreeSpool(self, actionName, inputValue, callbackState, isAnalog)
    local spec = self.spec_freeSpoolWinch
    if spec == nil then
        return
    end

    local now = g_time or 0
    if now - spec.lastToggleTime < spec.toggleDelay then
        return
    end
    spec.lastToggleTime = now

    self:setFreeSpoolEnabled(not spec.enabled)
end

-- API ------------------------------------------------------------------------

function FreeSpoolWinch:setFreeSpoolEnabled(state)
    local spec = self.spec_freeSpoolWinch
    if spec == nil then
        return
    end

    state = state and true or false
    if spec.enabled == state then
        return
    end

    spec.enabled = state

    -- Reset tracking when toggling off
    if not spec.enabled then
        spec.freeSpool = {}
    end

    if self.isClient then
        FreeSpoolWinch.updateActionEventText(self)

        local keyEnabled =
            g_i18n:hasText("info_winch_freeSpool_enabled") and "info_winch_freeSpool_enabled"
            or g_i18n:hasText("info_winch_freespool_enable") and "info_winch_freespool_enable"
            or nil

        local keyDisabled =
            g_i18n:hasText("info_winch_freeSpool_disabled") and "info_winch_freeSpool_disabled"
            or g_i18n:hasText("info_winch_freespool_disabled") and "info_winch_freespool_disabled"
            or nil

        local msg
        if spec.enabled then
            msg = keyEnabled and g_i18n:getText(keyEnabled) or "Winch Free-Spool Enabled"
        else
            msg = keyDisabled and g_i18n:getText(keyDisabled) or "Winch Free-Spool Disabled"
        end

        g_currentMission:showBlinkingWarning(msg, 1500)
    end
end

function FreeSpoolWinch:getFreeSpoolEnabled()
    local spec = self.spec_freeSpoolWinch
    return spec ~= nil and spec.enabled
end

-- Attachment target helper ---------------------------------------------------
-- Try to support:
--  - Base game trees      -> rope.attachedTrees[1].activeHookData:getRopeTargetPosition()
--  - BetterWinch-style    -> rope.attachedObjects / rope.attachedVehicle / custom data
-- Falls back gracefully if fields don't exist.

function FreeSpoolWinch.getAttachmentTarget(rope)
    -- Base game: attachedTrees
    if rope.attachedTrees ~= nil and #rope.attachedTrees > 0 then
        local a = rope.attachedTrees[1]
        if a.activeHookData ~= nil and a.activeHookData.getRopeTargetPosition ~= nil then
            return true, a.activeHookData:getRopeTargetPosition()
        end
    end

    -- Generic attachedObjects (common pattern in "attach anything" winch mods)
    if rope.attachedObjects ~= nil and #rope.attachedObjects > 0 then
        local o = rope.attachedObjects[1]

        -- Explicit rope target method on object
        if o.getRopeTargetPosition ~= nil then
            local x, y, z = o:getRopeTargetPosition()
            return true, x, y, z
        end

        -- Or a stored node id
        if o.node ~= nil and entityExists(o.node) then
            local x, y, z = getWorldTranslation(o.node)
            return true, x, y, z
        end
    end

    -- Single attachedVehicle pattern
    if rope.attachedVehicle ~= nil then
        local v = rope.attachedVehicle
        if v.getRopeTargetPosition ~= nil then
            local x, y, z = v:getRopeTargetPosition()
            return true, x, y, z
        end
        if v.rootNode ~= nil and entityExists(v.rootNode) then
            local x, y, z = getWorldTranslation(v.rootNode)
            return true, x, y, z
        end
    end

    -- If none matched, no valid target
    return false, 0, 0, 0
end

-- RUNTIME LOGIC: tension-based free spool (server-side) ----------------------

function FreeSpoolWinch:onPostUpdate(dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
    local spec = self.spec_freeSpoolWinch
    local winchSpec = self.spec_winch

    if spec == nil or winchSpec == nil then
        return
    end
    if not spec.enabled or not self.isServer then
        return
    end

    local slack = spec.slackDistance or 0.15
    local speedFactor = spec.speedFactor or 1.0

    for i, rope in ipairs(winchSpec.ropes) do
        -- Need an actual rope
        if rope.mainRope == nil or rope.ropeNode == nil then
            spec.freeSpool[i] = nil
        else
            local hasTarget, hx, hy, hz = FreeSpoolWinch.getAttachmentTarget(rope)

            if not hasTarget then
                -- Nothing attached (tree or object) -> no free spool on this rope
                spec.freeSpool[i] = nil
            else
                local rx, ry, rz = getWorldTranslation(rope.ropeNode)
                local dist = MathUtil.vector3Length(hx - rx, hy - ry, hz - rz)

                local fs = spec.freeSpool[i]
                if fs == nil then
                    -- Initialize allowed length at current distance so we don't insta-dump
                    fs = { allowedLength = dist }
                    spec.freeSpool[i] = fs
                end

                local allowed = fs.allowedLength or dist

                -- Only let out rope if something is pulling it beyond allowed + slack
                if dist > allowed + slack then
                    local ropeSpeed = rope.speed or 0.001
                    local delta = ropeSpeed * speedFactor * dt

                    -- Clamp against maxLength if present
                    local maxLen = rope.maxLength or math.huge
                    if allowed + delta > maxLen then
                        delta = math.max(0, maxLen - allowed)
                    end

                    if delta > 0 then
                        local changed = rope.mainRope:adjustLength(delta)

                        if changed ~= 0 then
                            fs.allowedLength = allowed + changed

                            -- Mark as "releasing" for FX/network coherence
                            rope.controlDirection = -1
                            rope.lastControlTimer = 100

                            if winchSpec.dirtyFlag ~= nil then
                                self:raiseDirtyFlags(winchSpec.dirtyFlag)
                            end
                            if winchSpec.ropeDirtyFlag ~= nil then
                                self:raiseDirtyFlags(winchSpec.ropeDirtyFlag)
                            end
                        end
                    end
                end
            end
        end
    end
end
