FreeSpoolWinch = {}
local FreeSpoolWinch_mt = Class(FreeSpoolWinch)

function FreeSpoolWinch.prerequisitesPresent(specializations)
    -- Require base-game Winch specialization
    return SpecializationUtil.hasSpecialization(Winch, specializations)
end

function FreeSpoolWinch.initSpecialization()
    local schema = Vehicle.xmlSchema
    schema:setXMLSpecializationType("FreeSpoolWinch")
    schema:register(XMLValueType.FLOAT, "vehicle.freeSpoolWinch#freeSpoolSpeedFactor", "Multiplier for free spool speed", 1.0)
    schema:setXMLSpecializationType()
end

function FreeSpoolWinch.registerFunctions(vehicleType)
    SpecializationUtil.registerFunction(vehicleType, "setFreeSpoolEnabled", FreeSpoolWinch.setFreeSpoolEnabled)
    SpecializationUtil.registerFunction(vehicleType, "getFreeSpoolEnabled", FreeSpoolWinch.getFreeSpoolEnabled)
end

function FreeSpoolWinch.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onLoad", FreeSpoolWinch)
    SpecializationUtil.registerEventListener(vehicleType, "onRegisterActionEvents", FreeSpoolWinch)
    SpecializationUtil.registerEventListener(vehicleType, "onPostUpdate", FreeSpoolWinch)
end

-- LOAD -----------------------------------------------------------------------

function FreeSpoolWinch:onLoad(savegame)
    local spec = {}
    self.spec_freeSpoolWinch = spec

    spec.enabled = false
    spec.lastToggleTime = 0
    spec.toggleDelay = 250 -- ms debounce
    spec.speedFactor = self.xmlFile:getValue("vehicle.freeSpoolWinch#freeSpoolSpeedFactor", 1.0)
end

-- INPUT / UI -----------------------------------------------------------------

function FreeSpoolWinch:onRegisterActionEvents(isActiveForInput, isActiveForInputIgnoreSelection)
    if not self.isClient then
        return
    end

    local winchSpec = self.spec_winch
    local spec = self.spec_freeSpoolWinch
    if winchSpec == nil or winchSpec.ropes == nil or #winchSpec.ropes == 0 then
        return
    end

    self:clearActionEventsTable(spec.actionEvents)

    if isActiveForInputIgnoreSelection then
        local _, actionEventId = self:addActionEvent(
            spec.actionEvents,
            InputAction.WINCH_FREESPOOL,
            self,
            FreeSpoolWinch.actionEventToggleFreeSpool,
            false, true, false, true, nil
        )
        g_inputBinding:setActionEventTextPriority(actionEventId, GS_PRIO_VERY_HIGH)
        FreeSpoolWinch.updateActionEventText(self)
    end
end

function FreeSpoolWinch.updateActionEventText(self)
    local spec = self.spec_freeSpoolWinch
    if not self.isClient or spec.actionEvents == nil then
        return
    end

    local ae = spec.actionEvents[InputAction.WINCH_FREESPOOL]
    if ae ~= nil then
        local text = spec.enabled and g_i18n:getText("action_winch_freeSpool_off") or g_i18n:getText("action_winch_freeSpool_on")
        g_inputBinding:setActionEventText(ae.actionEventId, text)
    end
end

function FreeSpoolWinch.actionEventToggleFreeSpool(self, actionName, inputValue, callbackState, isAnalog)
    local spec = self.spec_freeSpoolWinch
    if spec == nil then return end

    local now = g_time or 0
    if now - spec.lastToggleTime < spec.toggleDelay then
        return
    end
    spec.lastToggleTime = now

    self:setFreeSpoolEnabled(not spec.enabled)
end

-- API ------------------------------------------------------------------------

function FreeSpoolWinch:setFreeSpoolEnabled(state)
    local spec = self.spec_freeSpoolWinch
    if spec == nil then return end

    state = state and true or false
    if spec.enabled == state then
        return
    end

    spec.enabled = state

    if self.isServer then
        -- You could sync via dirtyFlag if you want MP support; minimal example skips that.
    end

    if self.isClient then
        FreeSpoolWinch.updateActionEventText(self)
        if spec.enabled then
            g_currentMission:showBlinkingWarning(g_i18n:getText("info_winch_freeSpool_enabled"), 1500)
        else
            g_currentMission:showBlinkingWarning(g_i18n:getText("info_winch_freeSpool_disabled"), 1500)
        end
    end
end

function FreeSpoolWinch:getFreeSpoolEnabled()
    local spec = self.spec_freeSpoolWinch
    return spec ~= nil and spec.enabled
end

-- RUNTIME LOGIC --------------------------------------------------------------

function FreeSpoolWinch:onPostUpdate(dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
    local spec = self.spec_freeSpoolWinch
    local winchSpec = self.spec_winch

    if spec == nil or winchSpec == nil or not spec.enabled or not self.isServer then
        return
    end

    -- For each rope: if we have at least one attached tree, let the rope out freely
    for _, rope in ipairs(winchSpec.ropes) do
        if #rope.attachedTrees > 0 and rope.mainRope ~= nil then
            local delta = rope.speed * spec.speedFactor * g_currentDt
            if delta > 0 then
                local changed = rope.mainRope:adjustLength(delta)
                if changed ~= 0 then
                    -- Mark as "releasing" for sounds/anim consistency with base Winch
                    rope.controlDirection = -1
                    rope.lastControlTimer = 100
                    self:raiseDirtyFlags(winchSpec.dirtyFlag)
                    self:raiseDirtyFlags(winchSpec.ropeDirtyFlag)
                end
            end
        end
    end
end
