--[[
    RegisterSpecialization

    Registers a specialization for a vehicle type. The script checks the prerequisites for the specialization and adds it to the vehicle type if it is not already present.

	@author: 		BayernGamers
	@date: 			05.12.2024
	@version:		1.1

	History:		v1.0 @12.11.2024 - initial implementation in FS25
                    ------------------------------------------------------------------------------------------------------
                    v1.1 @05.12.2024 - adjusted the script to be conform with the ModHub guidelines

	
	License:        Terms:
                        Usage:
                            Feel free to use this work as-is as long as you adhere to the following terms:
						Attribution:
							You must give appropriate credit to the original author when using this work.
						No Derivatives:
							You may not alter, transform, or build upon this work in any way.
						Usage: 
							The work may be used for personal and commercial purposes, provided it is not modified or adapted.
						Additional Clause:
							This script may not be converted, adapted, or incorporated into any other game versions or platforms except by GIANTS Software.
]]

source(Utils.getFilename("scripts/utils/LoggingUtil.lua", g_currentModDirectory))

local log = LoggingUtil.new(true, LoggingUtil.DEBUG_LEVELS.HIGH, "RegisterSpecialization.lua")

RegisterSpecialization = {}
RegisterSpecialization.debugLevelHigh = 2
RegisterSpecialization.debugLevel = 1
RegisterSpecialization.MOD_DIRECTORY = g_currentModDirectory
RegisterSpecialization.MOD_NAME = g_currentModName
RegisterSpecialization.XML_KEY = "modDesc.registerSpecializations.specialization"
RegisterSpecialization.modDescPath = RegisterSpecialization.MOD_DIRECTORY .. "modDesc.xml"
RegisterSpecialization.specializations = {}

function RegisterSpecialization.getClassObject(className)
    local parts = string.split(className, ".")
    local currentTable = _G[parts[1]]

    if type(currentTable) ~= "table" then
        return nil
    end

    for i = 2, #parts do
        currentTable = currentTable[parts[i]]

        if type(currentTable) ~= "table" then
            return nil
        end
    end

    return currentTable
end

local modDesc = XMLFile.load("ModFile", RegisterSpecialization.modDescPath)

modDesc:iterate(RegisterSpecialization.XML_KEY, function (_, key)
    log:printDevInfo("Loading Specialization data for key '" .. key .. "'", RegisterSpecialization.debugLevelHigh)
    local specName = modDesc:getString(key .. "#name")
    local typeName = specName
    local className = modDesc:getString(key .. "#className")
    local fileName = modDesc:getString(key .. "#fileName")

    if specName == nil or specName == "" then
        log:printXMLError(modDesc, "Missing value for 'name' in specialization" .. key)
    end

    if className == nil or className == "" then
        log:printXMLError(modDesc, "Missing value for 'className' in specialization" .. key)
    end

    if fileName == nil or fileName == "" then
        log:printXMLError(modDesc, "Missing value for 'fileName' in specialization" .. key)
    end

    local filePath = Utils.getFilename(fileName, RegisterSpecialization.MOD_DIRECTORY)

    if filePath == nil or filePath == "" then
        log:printXMLError(modDesc, "Invalid value for 'fileName' in specialization" .. key)
    end

    source(filePath)
    local class = RegisterSpecialization.getClassObject(className)

    if class == nil then
        log:printError("Class '" .. className .. "' not found in file '" .. fileName .. "'")
    end

    if class.prerequisitesPresent == nil then
        log:printError("Function 'prerequisitesPresent' not found in class '" .. className .. "'")
    end
    
    if g_specializationManager:getSpecializationByName(specName) == nil then
        g_specializationManager:addSpecialization(typeName, className, filePath, nil)
        log:printDevInfo("Added specialization '" .. typeName .. "'", RegisterSpecialization.debugLevelHigh)
    end

    for vehicleTypeName, vehicleType in pairs(g_vehicleTypeManager.types) do
        if vehicleType ~= nil and class.prerequisitesPresent(vehicleType.specializations) then
            g_vehicleTypeManager:addSpecialization(vehicleTypeName, typeName)
            log:printDevInfo("Added specialization '" .. typeName .. "' to " .. "'" .. vehicleTypeName .. "'", RegisterSpecialization.debugLevelHigh)
        end
    end
end)