-- Copyright 2024 by Todd Hundersmarck (ThundR)
-- All Rights Reserved

local thMain = g_thPlaceableDesignKit
local thModManager = thMain.thModManager
local thI18n = thMain.i18n
THPlaceableDesignSpec = {}
THPlaceableDesignSpec.MOD_NAME = thMain.modName
THPlaceableDesignSpec.MOD_PATH = thMain.modPath
THPlaceableDesignSpec.SPEC_KEY = THPlaceableDesignKit.DATA_KEY
THPlaceableDesignSpec.XML_KEY = THPlaceableDesignSpec.SPEC_KEY
THPlaceableDesignSpec.CONFIG_XML_KEY = "placeable." .. THPlaceableDesignSpec.XML_KEY
THPlaceableDesignSpec.CONFIG_XML_PATH = THPlaceableDesignSpec.CONFIG_XML_KEY
THPlaceableDesignSpec.SAVEGAME_XML_PATH = "placeables.placeable(?)." .. THPlaceableDesignSpec.MOD_NAME .. "." .. THPlaceableDesignSpec.XML_KEY
THPlaceableDesignSpec.CONFIG_TYPE = {
    DESIGN = 1,
    COLOR = 2
}
THPlaceableDesignSpec.configTypes = {
    { id = "DESIGN", index = 1, typeClass = PlaceableConfigurationItem },
    { id = "COLOR", index = 2, typeClass = PlaceableConfigurationItemColor }
}
THPlaceableDesignSpec.configurations = {}
THPlaceableDesignSpec.configMapping = {}
THPlaceableDesignSpecTable = {}
local function getSpecTable(self)
    return THUtils.getSpecTable(self, THPlaceableDesignSpec.SPEC_KEY, THPlaceableDesignSpec.MOD_NAME)
end
THPlaceableDesignSpec.getSpecTable = getSpecTable
function THPlaceableDesignSpec.initSpecialization()
    THUtils.call(function()
        if THGameVersion == 25 then
            local configManager = g_placeableConfigurationManager
            local function addConfig(pConfigName, pConfigType)
                local configTypeData = THPlaceableDesignSpec.configTypes[pConfigType]
                if THUtils.argIsValid(configTypeData ~= nil, "configType", pConfigType) then
                    local configInfo = configManager:getConfigurationDescByName(pConfigName)
                    if configInfo ~= nil and THPlaceableDesignSpec.configMapping[configInfo.name] == nil then
                        local configData = {
                            name = configInfo.name,
                            type = configTypeData.index,
                            info = configInfo
                        }
                        table.insert(THPlaceableDesignSpec.configurations, configData)
                        THPlaceableDesignSpec.configMapping[configData.name] = #THPlaceableDesignSpec.configurations
                    end
                end
            end
            local modsArray, numMods = thModManager:getLoadedMods()
            if numMods > 0 then
                for _, modData in ipairs(modsArray) do
                    local modXMLFile = THUtils.loadXMLFile("THModDesc", modData.file, true)
                    if modXMLFile ~= nil then
                        local configsKey = string.format("modDesc.%s.configurations", thMain.dataKey)
                        if THUtils.hasXMLProperty(modXMLFile, configsKey) then
                            modXMLFile:iterate(configsKey .. ".config", function(_, pConfigKey)
                                local configName = modXMLFile:getString(pConfigKey .. "#name")
                                local configTitle = modXMLFile:getString(pConfigKey .. "#title")
                                local configTypeId = modXMLFile:getString(pConfigKey .. "#type")
                                local configTypeData = nil
                                if configTypeId ~= nil then
                                    local configTypeIndex = THPlaceableDesignSpec.CONFIG_TYPE[configTypeId:upper()]
                                    configTypeData = THPlaceableDesignSpec.configTypes[configTypeIndex]
                                end
                                local configPathText = string.format("%s: %s", modData.name, pConfigKey)
                                if configTypeData == nil then
                                    THUtils.xmlErrorMsg(configPathText, nil, THMessage.INVALID_VALUE, "type", configTypeId)
                                elseif not THUtils.validateId(configName) then
                                    THUtils.xmlErrorMsg(configPathText, nil, THMessage.INVALID_VALUE, "name", configName)
                                else
                                    local configInfo = configManager:getConfigurationDescByName(configName)
                                    if configInfo == nil then
                                        if configTitle == nil then
                                            configTitle = configName
                                        else
                                            configTitle = thI18n:convertText(configTitle, modData.name)
                                        end
                                        configManager:addConfigurationType(configName, configTitle, THPlaceableDesignSpec.XML_KEY, configTypeData.typeClass)
                                        addConfig(configName, configTypeData.index)
                                    end
                                end
                            end)
                        end
                        THUtils.deleteXMLFile(modXMLFile)
                    end
                end
            end
        end
    end)
end
function THPlaceableDesignSpec.prerequisitesPresent(specializations)
    return true
end
function THPlaceableDesignSpec.registerXMLPaths(xmlSchema, xmlPath)
    xmlSchema:setXMLSpecializationType("THPlaceableDesign")
    THUtils.call(function()
        local xmlBasePath = THPlaceableDesignSpec.CONFIG_XML_PATH
        local sharedColorPath = xmlBasePath .. ".sharedColors.color(?)"
        THUtils.registerXMLPath(xmlSchema, XMLValueType.STRING, sharedColorPath .. "#name", "Shared color name")
        THUtils.registerXMLPath(xmlSchema, XMLValueType.COLOR, sharedColorPath .. "#color", "Shared color value")
    end)
    xmlSchema:setXMLSpecializationType()
end
function THPlaceableDesignSpec.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onLoad", THPlaceableDesignSpec)
end
function THPlaceableDesignSpec.onLoad(self, savegame)
    local specTable = getSpecTable(self)
    specTable.parent = self
    THUtils.call(function()
        THUtils.copyFunctions(THPlaceableDesignSpecTable, specTable, false)
        local xmlFile = self.xmlFile
        local xmlBaseKey = THPlaceableDesignSpec.CONFIG_XML_KEY
        local components = self.components
        local i3dMappings = self.i3dMappings
        if xmlFile ~= nil then
            specTable:loadPlaceableDesignData(xmlFile, xmlBaseKey, components, i3dMappings)
        end
        specTable.areHooksActive = true
        if not specTable.areHooksCreated then
            specTable.areHooksCreated = true
        end
    end)
end
function THPlaceableDesignSpec.inj_setShaderParameter(superFunc, nodeId, paramName, r, g, b, a, ...)
    if THPlaceableDesignSpec.nodeParams ~= nil
        and paramName == "colorScale0"
    then
        THUtils.call(function()
            local nodePath = I3DUtil.getNodePath(nodeId)
            local newParamName = THPlaceableDesignSpec.nodeParams[nodePath]
            if newParamName ~= nil and newParamName ~= paramName then
                paramName = newParamName
            end
        end)
    end
    return superFunc(nodeId, paramName, r, g, b, a, ...)
end
function THPlaceableDesignSpec.inj_registerColorXMLPaths(superFunc, xmlSchema, xmlRootPath, xmlPath, ...)
    local function appendFunc(...)
        if xmlSchema ~= nil and xmlRootPath ~= nil then
            THUtils.call(function()
                local xmlSubPath = xmlRootPath .. "." .. THPlaceableDesignSpec.XML_KEY
                local nodeParamPath = xmlSubPath .. ".nodeParams.node(?)"
                THUtils.registerXMLPath(xmlSchema, XMLValueType.NODE_INDEX, nodeParamPath .. "#node", "Node id")
                THUtils.registerXMLPath(xmlSchema, XMLValueType.STRING, nodeParamPath .. "#shader", "Shader parameter")
            end)
        end
        return ...
    end
    return appendFunc(superFunc(xmlSchema, xmlRootPath, xmlPath, ...))
end
function THPlaceableDesignSpec.inj_onPostLoadColor(superFunc, item, placeable, ...)
    if placeable ~= nil then
        THUtils.call(function()
            local manager = g_placeableConfigurationManager
            local xmlFile = placeable.xmlFile
            local components = placeable.components
            local i3dMappings = placeable.i3dMappings
            if xmlFile ~= nil then
                local itemData = THUtils.getDataTable(item, thMain)
                if itemData == nil then
                    itemData = THUtils.createDataTable(item, thMain)
                    if itemData == nil then
                        return
                    end
                    itemData.nodeParams = {}
                    local configInfo = manager:getConfigurationDescByName(item.configName)
                    if configInfo ~= nil and configInfo.configurationsKey ~= nil then
                        local xmlSubKey = configInfo.configurationsKey .. "." .. THPlaceableDesignSpec.XML_KEY
                        xmlFile:iterate(xmlSubKey .. ".nodeParams.node", function(_, pNodeKey)
                            local nodeId = THUtils.getXMLValue(xmlFile, pNodeKey .. "#node", nil, components, i3dMappings)
                            local shaderParam = THUtils.getXMLValue(xmlFile, pNodeKey .. "#shader")
                            if THUtils.getNoNil(nodeId, 0) <= 0 then
                                THUtils.xmlErrorMsg(pNodeKey, nil, THMessage.INVALID_VALUE, "node", nodeId)
                            elseif shaderParam == nil or not getHasShaderParameter(nodeId, shaderParam) then
                                THUtils.xmlErrorMsg(pNodeKey, nil, THMessage.INVALID_VALUE, "shader", shaderParam)
                            else
                                local nodePath = I3DUtil.getNodePath(nodeId)
                                if nodePath ~= nil then
                                    itemData.nodeParams[nodePath] = shaderParam
                                end
                            end
                        end)
                    end
                end
                if itemData ~= nil then
                    THPlaceableDesignSpec.nodeParams = itemData.nodeParams
                end
            end
        end)
    end
    local function appendFunc(...)
        THPlaceableDesignSpec.nodeParams = nil
        return ...
    end
    return appendFunc(superFunc(item, placeable, ...))
end
THUtils.call(function()
    THUtils.setFunctionHook(g_thGlobalEnv, "setShaderParameter", false, false, nil, THPlaceableDesignSpec.inj_setShaderParameter)
    THUtils.setFunctionHook("PlaceableConfigurationItemColor", "registerXMLPaths", false, false, nil, THPlaceableDesignSpec.inj_registerColorXMLPaths)
    THUtils.setFunctionHook("PlaceableConfigurationItemColor", "onPostLoad", false, false, nil, THPlaceableDesignSpec.inj_onPostLoadColor)
end)
function THPlaceableDesignSpecTable.loadPlaceableDesignData(self, xmlFile, xmlKey, components, i3dMappings, customEnv, basePath)
    customEnv = customEnv or THPlaceableDesignSpec.MOD_NAME
    basePath = basePath or THPlaceableDesignSpec.MOD_PATH
    local success = false
    if THUtils.argIsValid(type(xmlFile) == "table", "xmlFile", xmlFile)
        and THUtils.argIsValid(type(xmlKey) == "string", "xmlKey", xmlKey)
    then
        success = true
    end
    return success
end