-- Copyright 2024-2025 by Todd Hundersmarck (ThundR) 
-- All Rights Reserved

_G.THDebugLevel = {
    CRITICAL = 1,
    ERROR = 2,
    WARNING = 3,
    NORMAL = 4,
    UPDATE = 5,
    LOOP = 6
}
THUtils.createEnumTable(THDebugLevel)
_G.debugMsg = printf
THSetDebugEnabledEvent = {}
local THSetDebugEnabledEvent_mt = Class(THSetDebugEnabledEvent, Event)
THSetDebugLevelEvent = {}
local THSetDebugLevelEvent_mt = Class(THSetDebugLevelEvent, Event)
THUtils.initClass(THSetDebugEnabledEvent, "THSetDebugEnabledEvent", THClassType.EVENT)
THUtils.initClass(THSetDebugLevelEvent, "THSetDebugLevelEvent", THClassType.EVENT)
THDebugUtil = {}
THDebugUtil.debugFlagIds = {}
THDebugUtil.globalDebugFlag = false
THDebugUtil.currentDebugLevel = THDebugLevel.NORMAL
local function initScript()
    if g_thDebugTools ~= nil then
        THDebugUtil.setIsEnabled()
    end
end
function THDebugUtil.getFlagId(flagId)
    if flagId == nil then
        flagId = ""
    elseif type(flagId) == THValueType.STRING then
        if flagId ~= "" then
            flagId = string.upper(flagId)
        end
    else
        THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "flagId", flagId)
        return
    end
    if flagId == "" or THDebugUtil.debugFlagIds[flagId] ~= nil then
        return flagId
    end
    THUtils.errorMsg(nil, "Debug flag %q does not exist", flagId)
end
function THDebugUtil.getFlagIds()
    return THDebugUtil.debugFlagIds
end
function THDebugUtil.getDebugLevel()
    return THDebugUtil.currentDebugLevel
end
function THDebugUtil.getDebugLevelById(debugLevelId)
    if debugLevelId ~= nil then
        local idType = type(debugLevelId)
        local debugLevel = nil
        if idType == THValueType.STRING then
            debugLevel = THDebugLevel:getValue(debugLevelId)
            if debugLevel == nil then
                debugLevel = THUtils.toNumber(debugLevelId, true)
                if debugLevel ~= nil then
                    if THDebugLevel:getId(debugLevel) == nil then
                        debugLevel = nil
                    end
                end
            end
        elseif idType == THValueType.NUMBER then
            if THDebugLevel:getId(debugLevelId) ~= nil then
                debugLevel = debugLevelId
            end
        else
            THUtils.errorMsg(nil, THMessage.INVALID_VALUE, "debugLevelId", debugLevelId)
        end
        return debugLevel
    end
end
function THDebugUtil.setDebugLevel(debugLevelId)
    local debugLevel = THDebugUtil.getDebugLevelById(debugLevelId)
    local success = false
    if debugLevel == nil then
        THUtils.errorMsg(nil, THMessage.INVALID_VALUE, "debugLevelId", debugLevelId)
    else
        THUtils.displayMsg("Debug level set to: %s", debugLevelId)
        THDebugUtil.currentDebugLevel = debugLevel
        success = true
    end
    return success, THDebugUtil.currentDebugLevel
end
function THDebugUtil.createFlagId(flagName, isEnabled)
    isEnabled = THUtils.validateArg(not isEnabled or isEnabled == true, "isEnabled", isEnabled, false)
    if THUtils.argIsValid(THUtils.validateId(flagName), "flagName", flagName) then
        local flagId = flagName:upper()
        if THDebugUtil.debugFlagIds[flagId] == nil then
            THDebugUtil.debugFlagIds[flagId] = isEnabled == true
        end
        return flagId
    end
end
function THDebugUtil.getIsEnabled(flagName, debugLevelId)
    local reqDebugLevel = nil
    if debugLevelId == nil then
        reqDebugLevel = THDebugUtil.currentDebugLevel
    else
        reqDebugLevel = THDebugUtil.getDebugLevelById(debugLevelId)
    end
    local flagId = THDebugUtil.getFlagId(flagName)
    local isDebugEnabled = false
    if flagId ~= nil
        and reqDebugLevel ~= nil and reqDebugLevel <= THDebugUtil.currentDebugLevel
    then
        if flagId == "" then
            isDebugEnabled = THDebugUtil.globalDebugFlag == true
        else
            isDebugEnabled = THDebugUtil.debugFlagIds[flagId] == true
        end
    end
    return isDebugEnabled, flagId, reqDebugLevel
end
function THDebugUtil.setIsEnabled(flagName, isEnabled)
    if type(flagName) == THValueType.BOOLEAN then
        isEnabled = flagName
        flagName = nil
    end
    local flagId = THDebugUtil.getFlagId(flagName)
    if flagId ~= nil then
        isEnabled = THUtils.getNoNil(isEnabled, true)
        if THUtils.argIsValid(not isEnabled or isEnabled == true, "isEnabled", isEnabled) then
            local newIsEnabled = nil
            if flagId == "" then
                THDebugUtil.globalDebugFlag = isEnabled == true
                newIsEnabled = THDebugUtil.globalDebugFlag
            else
                THDebugUtil.debugFlagIds[flagId] = isEnabled == true
                newIsEnabled = THDebugUtil.debugFlagIds[flagId]
            end
            if newIsEnabled == nil then
                if flagId == "" then
                    THUtils.errorMsg(true, "Failed to set global debug flag")
                else
                    THUtils.errorMsg(true, "Failed to set debug flag: %s", flagName)
                end
            else
                local enabledText = "disabled"
                if newIsEnabled == true then
                    enabledText = "enabled"
                end
                if flagId == "" then
                    THUtils.displayMsg("Debugger is %s", enabledText)
                else
                    THUtils.displayMsg("Debugging for %q is %s", flagId, enabledText)
                end
                return true, newIsEnabled, flagId
            end
        end
    end
    return false
end
function THDebugUtil.resetDebug()
    THDebugUtil.setIsEnabled(nil, false)
    THDebugUtil.setDebugLevel(THDebugLevel.NORMAL)
end
function THDebugUtil.printMsg(flagName, debugLevelId, text, ...)
    local isDebugEnabled, _, debugLevel = THDebugUtil.getIsEnabled(flagName, debugLevelId)
    if isDebugEnabled then
        if debugLevel == THDebugLevel.CRITICAL then
            THUtils.errorMsg(true, text, ...)
        elseif debugLevel == THDebugLevel.ERROR then
            THUtils.errorMsg(nil, text, ...)
        elseif debugLevel == THDebugLevel.WARNING then
            THUtils.errorMsg(false, text, ...)
        else
            THUtils.displayMsg(text, ...)
        end
    end
end
function THDebugUtil.printError(flagName, showStack, verbose, text, ...)
    showStack = THUtils.validateArg(not showStack or showStack == true, "showStack", showStack, true)
    verbose = THUtils.validateArg(not verbose or verbose == true, "verbose", verbose, false)
    if verbose then
        THUtils.errorMsg(showStack, text, ...)
    else
        local debugLevel = THDebugLevel.ERROR
        if showStack == false then
            debugLevel = THDebugLevel.WARNING
        elseif showStack == true then
            debugLevel = THDebugLevel.CRITICAL
        end
        THDebugUtil.printMsg(flagName, debugLevel, text, ...)
    end
end
function THDebugUtil.printTable(target, ...)
    if g_thDebugTools ~= nil then
        return g_thDebugTools:printTable(target, ...)
    end
    local targetName = target
    if type(target) == THValueType.STRING then
        target = THUtils.getTableValue(nil, targetName)
    end
    if THUtils.argIsValid(type(target) == THValueType.TABLE, "target", targetName) then
        THUtils.displayMsg("Printing %s", targetName)
        for key, value in pairs(target) do
            THUtils.displayMsg("[%s]: %s", key, value)
        end
    end
end
THDebugUtil.dumpTable = THDebugUtil.printTable
function THDebugUtil.printTableValues(target)
    if THUtils.argIsValid(type(target) == THValueType.TABLE, "target", target) then
        THUtils.displayMsg("Printing %s ", target)
        for key, value in pairs(target) do
            local valueType = type(value)
            if valueType ~= THValueType.TABLE
                and valueType ~= THValueType.FUNCTION
            then
                THUtils.displayMsg("[%s]: %s", key, value)
            end
        end
        THUtils.displayMsg("")
    end
end
THDebugUtil.dumpTableValues = THDebugUtil.printTableValues
function THDebugUtil.traceFunction(...)
    if g_thDebugTools ~= nil then
        return g_thDebugTools:traceFunction(...)
    end
end
function THDebugUtil.createConsoleCommands(target)
    if THUtils.argIsValid(type(target) == THValueType.TABLE, "target", target) then
        if target.name == nil then
            THUtils.errorMsg(nil, "Cannot create console commands, no data key found for target")
        else
            local callbackTarget = THDebugUtil
            if target.consoleCommandSetDebugEnabled ~= nil then
                if type(target.consoleCommandSetDebugEnabled) ~= THValueType.FUNCTION then
                    THUtils.errorMsg(false, "Cannot create console command: %s", "consoleCommandSetDebugEnabled")
                else
                    callbackTarget = target
                end
            end
            addConsoleCommand(target.name .. ":setDebugEnabled", "Sets debug flagId enabled status", "consoleCommandSetDebugEnabled", callbackTarget, "[flagId] [isEnabled] [sendToServer]")
            callbackTarget = THDebugUtil
            if target.consoleCommandSetDebugLevel ~= nil then
                if type(target.consoleCommandSetDebugLevel) ~= THValueType.FUNCTION then
                    THUtils.errorMsg(false, "Cannot create console command: %s", "consoleCommandSetDebugLevel")
                else
                    callbackTarget = target
                end
            end
            addConsoleCommand(target.name .. ":setDebugLevel", "Sets debug level", "consoleCommandSetDebugLevel", callbackTarget, "debugLevelId [sendToServer]")
        end
    end
end
function THDebugUtil.consoleCommandSetDebugEnabled(flagName, isEnabled, sendToServer)
    THUtils.call(function()
        local usageMsg = "Usage: setDebugEnabled [flagId] [isEnabled] [sendToServer]"
        local function showUsageMessage(pShowFlagList)
            printf(usageMsg)
            if pShowFlagList == true then
                printf("")
                printf("Allowed debug flagIds:")
                local debugFlagIds = THDebugUtil.getFlagIds()
                for otherFlagId in pairs(debugFlagIds) do
                    printf("- %s", otherFlagId)
                end
                printf("")
            end
        end
        local flagNameBool = THUtils.toBoolean(flagName)
        local flagId = nil
        if flagNameBool ~= nil then
            flagId = ""
            isEnabled = flagNameBool
            sendToServer = string.upper(tostring(isEnabled))
        else
            flagId = THDebugUtil.getFlagId(flagName)
            isEnabled = string.upper(tostring(isEnabled))
            sendToServer = string.upper(tostring(sendToServer))
        end
        if flagId == nil then
            showUsageMessage(true)
        else
            if isEnabled == "" or isEnabled == "NIL" or isEnabled == "TRUE" then
                isEnabled = true
            elseif isEnabled == "FALSE" then
                isEnabled = false
            else
                showUsageMessage(false)
                return
            end
            if sendToServer == "" or sendToServer == "NIL" or sendToServer == "FALSE" then
                sendToServer = false
            elseif sendToServer == "TRUE" then
                sendToServer = true
            else
                showUsageMessage(false)
                return
            end
            local success, newIsEnabled = THDebugUtil.setIsEnabled(flagId, isEnabled == true)
            if success and g_server == nil
                and sendToServer == true
            then
                THSetDebugEnabledEvent.sendEvent(flagId, newIsEnabled == true)
            end
        end
    end)
end
function THDebugUtil.consoleCommandSetDebugLevel(self, debugLevelId, sendToServer)
    local usageMsg = "Usage: setDebugLevel debugLevel [sendToServer]"
    sendToServer = THUtils.toBoolean(sendToServer)
    local function showUsageMsg(pShowLevelList)
        printf(usageMsg)
        if pShowLevelList == true then
            printf("")
            printf("Allowed debug level ids:")
            local allowedValues = THDebugLevel:getValues()
            for id, level in pairs(allowedValues) do
                printf("[%s]: %s", id, level)
            end
            printf("")
        end
    end
    local debugLevel = THDebugUtil.getDebugLevelById(debugLevelId)
    if debugLevel == nil or debugLevel <= 0 then
        showUsageMsg(true)
        return
    end
    sendToServer = tostring(sendToServer):upper()
    if sendToServer == "" or sendToServer == "NIL" or sendToServer == "FALSE" then
        sendToServer = false
    elseif sendToServer == "TRUE" then
        sendToServer = true
    else
        THUtils.errorMsg(nil, THMessage.ARGUMENT_INVALID, "sendToServer", sendToServer)
        showUsageMsg(false)
        return
    end
    local success, newDebugLevel = THDebugUtil.setDebugLevel(debugLevel)
    if success and g_server == nil
        and sendToServer == true
    then
        THSetDebugLevelEvent.sendEvent(newDebugLevel)
    end
end
function THSetDebugEnabledEvent.emptyNew()
    return Event.new(THSetDebugEnabledEvent_mt)
end
function THSetDebugEnabledEvent.new(flagName, isEnabled)
    local self = THSetDebugEnabledEvent.emptyNew()
    self.isEventValid = false
    if THUtils.argIsValid(not isEnabled or isEnabled == true, "isEnabled", isEnabled) then
        local flagId = THDebugUtil.getFlagId(flagName)
        if flagId ~= nil then
            if g_server ~= nil then
                THUtils.errorMsg(true, THMessage.EVENT_NO_SERVER)
            elseif not THUtils.getIsMasterUser() then
                THUtils.errorMsg(false, THMessage.EVENT_ADMIN_ONLY)
            else
                self.flagId = flagId
                self.isEnabled = THUtils.getNoNil(isEnabled, true)
                self.isEventValid = true
            end
        end
    end
    return self
end
function THSetDebugEnabledEvent.readStream(self, streamId, connection)
    if not connection:getIsServer() then
        if streamReadBool(streamId) then
            self.flagId = streamReadString(streamId)
            self.isEnabled = streamReadBool(streamId)
            self:run(connection)
        end
    else
        THUtils.errorMsg(true, THMessage.EVENT_NO_CLIENT_READ)
    end
end
function THSetDebugEnabledEvent.writeStream(self, streamId, connection)
    if connection:getIsServer() then
        if streamWriteBool(streamId, self.isEventValid) then
            streamWriteString(streamId, self.flagId)
            streamWriteBool(streamId, self.isEnabled)
        end
    else
        THUtils.errorMsg(true, THMessage.EVENT_NO_SERVER_WRITE)
    end
end
function THSetDebugEnabledEvent.run(self, connection)
    if not connection:getIsServer() then
        if self.isEventValid then
            THDebugUtil.setIsEnabled(self.flagId, self.isEnabled)
        end
    end
end
function THSetDebugEnabledEvent.sendEvent(flagName, isEnabled)
    local success = false
    if g_server == nil and g_client ~= nil then
        local newEvent = THSetDebugEnabledEvent.new(flagName, isEnabled)
        if newEvent ~= nil and newEvent.isEventValid then
            g_client:getServerConnection():sendEvent(newEvent)
            success = true
        end
    end
    return success
end
function THSetDebugLevelEvent.emptyNew()
    return Event.new(THSetDebugLevelEvent_mt)
end
function THSetDebugLevelEvent.new(debugLevelId)
    local self = THSetDebugLevelEvent.emptyNew()
    local debugLevel = THDebugUtil.getDebugLevelById(debugLevelId)
    self.isEventValid = false
    if THUtils.argIsValid(debugLevel ~= nil, "debugLevelId", debugLevelId) then
        if g_server ~= nil then
            THUtils.errorMsg(true, THMessage.EVENT_NO_SERVER)
        elseif not THUtils.getIsMasterUser() then
            THUtils.errorMsg(false, THMessage.EVENT_ADMIN_ONLY)
        else
            self.debugLevel = math.floor(debugLevel)
            self.isEventValid = true
        end
    end
    return self
end
function THSetDebugLevelEvent.readStream(self, streamId, connection)
    if not connection:getIsServer() then
        if streamReadBool(streamId) then
            self.debugLevel = streamReadUIntN(streamId, 4)
            self:run(connection)
        end
    else
        THUtils.errorMsg(true, THMessage.EVENT_NO_CLIENT_READ)
    end
end
function THSetDebugLevelEvent.writeStream(self, streamId, connection)
    if connection:getIsServer() then
        if streamWriteBool(streamId, self.isEventValid) then
            streamWriteUIntN(streamId, self.debugLevel, 4)
        end
    else
        THUtils.errorMsg(true, THMessage.EVENT_NO_SERVER_WRITE)
    end
end
function THSetDebugLevelEvent.run(self, connection)
    if not connection:getIsServer() then
        if self.isEventValid then
            THDebugUtil.setDebugLevel(self.debugLevel)
        end
    end
end
function THSetDebugLevelEvent.sendEvent(debugLevelId)
    local success = false
    if g_server == nil and g_client ~= nil then
        local newEvent = THSetDebugLevelEvent.new(debugLevelId)
        if newEvent ~= nil and newEvent.isEventValid then
            g_client:getServerConnection():sendEvent(newEvent)
            success = true
        end
    end
    return success
end
THUtils.call(initScript)