-- Copyright 2024-2025 by Todd Hundersmarck (ThundR) 
-- All Rights Reserved

local thModName = g_currentModName
local thModPath = g_currentModDirectory
source(thModPath .. "scripts/core/THCore.lua")
THDefaultTypes = {}
local THDefaultTypes_mt = THUtils.createClass(THDefaultTypes, THCore)
THDefaultTypes.NAME = "thDefaultTypes"
THDefaultTypes.CLASS_NAME = "THDefaultTypes"
THDefaultTypes.MOD_NAME = thModName
THDefaultTypes.MESSAGE = {
    FRUIT_TYPE_REMOVED = "Removing fruit type: %q",
    FRUIT_TYPE_SKIPPED = "Skipping fruit type: %q",
    DUPLICATE_FRUIT_TYPE = "Duplicate fruit type found: %q, swapping with existing entry",
    FILL_TYPE_REMOVED = "Removing fill type: %q",
    FILL_TYPE_SKIPPED = "Skipping fill type: %q",
    RESTRICTED_FILL_TYPE = "Fill type %q is restricted and cannot be removed",
    DUPLICATE_FILL_TYPE = "Duplicate fill type found: %q, swapping with existing entry",
    HEIGHT_TYPE_REMOVED = "Removing density map height type: %q",
    HEIGHT_TYPE_SKIPPED = "Skipping density map height type: %q",
    RESTRICTED_HEIGHT_TYPE = "Density map height type %q is restricted and cannot be removed",
    BALE_REMOVED = "Removing bale: %s",
    BALE_SKIPPED = "Skipping bale: %s",
    STORE_ITEM_REMOVED = "Removing store item: %s",
    STORE_ITEM_SKIPPED = "Skipping store item: %s"
}
local debugFlagId = THDebugUtil.createFlagId("THDefaultTypes", true)
THDefaultTypes.debugFlagId = debugFlagId
local function initScript()
    local self = THDefaultTypes.new(THDefaultTypes.NAME, THDefaultTypes.CLASS_NAME, THDefaultTypes.MOD_NAME)
    if self ~= nil then
        _G.g_thDefaultTypes = self
    end
end
function THDefaultTypes.new(name, className, modName, customMt)
    customMt = customMt or THDefaultTypes_mt
    local self = THCore.new(name, className, modName, customMt)
    if self ~= nil then
        self.typeManagers = {
            fruitTypes = g_fruitTypeManager,
            fillTypes = g_fillTypeManager
        }
        self.typeConfig = {
            fruitTypes = {
                blackList = {}
            },
            fillTypes = {
                loadMapFirst = false,
                isMapLoadFinished = false,
                isDefaultTypesLoading = false,
                isDefaultTypesLoadFinished = false,
                areBaseTypesLoaded = false,
                blackList = {},
                restricted = {
                    AIR = true,
                    BALE = true,
                    CHAFF = true,
                    DEF = true,
                    DIESEL = true,
                    DIGESTATE = true,
                    DRYGRASS = true,
                    DRYGRASS_WINDROW = true,
                    ELECTRICCHARGE = true,
                    EGG = true,
                    FERTILIZER = true,
                    FORAGE = true,
                    FORAGE_MIXING = true,
                    GRASS = true,
                    GRASS_WINDROW = true,
                    HERBICIDE = true,
                    LIQUIDFERTILIZER = true,
                    LIQUIDMANURE = true,
                    MANURE = true,
                    MINERAL_FEED = true,
                    PIGFOOD = true,
                    MEADOW = true,
                    METHANE = true,
                    SILAGE = true,
                    SEEDS = true,
                    SNOW = true,
                    STONE = true,
                    STRAW = true,
                    TARP = true,
                    TREE = true,
                    WATER = true,
                    WEED = true,
                    WOOD = true,
                    WOODCHIPS = true,
                    WOOL = true
                }
            },
            heightTypes = {
                blackList = {},
                restricted = {
                    CHAFF = true,
                    DIGESTATE = true,
                    DRYGRASS = true,
                    DRYGRASS_WINDROW = true,
                    FERTILIZER = true,
                    FORAGE = true,
                    FORAGE_MIXING = true,
                    GRASS = true,
                    GRASS_WINDROW = true,
                    LIQUIDMANURE = true,
                    MANURE = true,
                    MINERAL_FEED = true,
                    PIGFOOD = true,
                    SILAGE = true,
                    SNOW = true,
                    STONE = true,
                    STRAW = true,
                    TARP = true,
                    WOODCHIPS = true
                },
            },
            bales = {
                blackList = {}
            },
            storeItems = {
                blackList = {}
            }
        }
        self.needsUpdateHerbicideAreaFix = false
        return self
    end
end
function THDefaultTypes.onPreInit(self, ...)
    THUtils.callSuperClass(self, "onPreInit", true, ...)
    THUtils.setFunctionHook(g_thGlobalEnv, "loadXMLFile", false, false, self, THDefaultTypes.inj_loadXMLFile)
    THUtils.setFunctionHook("XMLUtil", "loadDataFromMapXML", false, false, self, THDefaultTypes.inj_loadDataFromMapXML)
    THUtils.setFunctionHook("FruitTypeManager", "loadMapData", false, false, self, THDefaultTypes.inj_loadFruitTypeMapData)
    THUtils.setFunctionHook("FruitTypeManager", "loadFruitTypes", false, false, self, THDefaultTypes.inj_loadFruitTypes)
    THUtils.setFunctionHook("FruitTypeManager", "loadMapFruitTypes", false, false, self, THDefaultTypes.inj_loadMapFruitTypes)
    THUtils.setFunctionHook("FruitTypeManager", "loadCategoriesFromXML", false, false, self, THDefaultTypes.inj_loadFruitTypeCategoriesFromXML)
    THUtils.setFunctionHook("FruitTypeManager", "loadConvertersFromXML", false, false, self, THDefaultTypes.inj_loadFruitTypeConvertersFromXML)
    THUtils.setFunctionHook("FruitTypeDesc", "loadFromFoliageXMLFile", false, false, self, THDefaultTypes.inj_loadFruitTypeFromFoliageXMLFile)
    THUtils.setFunctionHook("WeedSystem", "loadReplacements", false, false, self, THDefaultTypes.inj_loadWeedSystemReplacements)
    THUtils.setFunctionHook("FillTypeManager", "loadMapData", false, false, self, THDefaultTypes.inj_loadFillTypeMapData)
    THUtils.setFunctionHook("FillTypeManager", "loadDefaultTypes", false, false, self, THDefaultTypes.inj_loadDefaultFillTypes)
    THUtils.setFunctionHook("FillTypeManager", "loadFillTypes", false, false, self, THDefaultTypes.inj_loadFillTypes)
    THUtils.setFunctionHook("FillTypeDesc", "loadFromXMLFile", false, false, self, THDefaultTypes.inj_loadFillTypeFromXMLFile)
    THUtils.setFunctionHook("DensityMapHeightManager", "loadDensityMapHeightTypes", false, false, self, THDefaultTypes.inj_loadDensityMapHeightTypes)
    THUtils.setFunctionHook("BaleManager", "loadBaleFromXML", false, false, self, THDefaultTypes.inj_loadBaleFromXML)
    THUtils.setFunctionHook("BaleManager", "loadModBaleFromXML", false, false, self, THDefaultTypes.inj_loadModBaleFromXML)
    THUtils.setFunctionHook("BaleManager", "loadBaleDataFromXML", false, false, self, THDefaultTypes.inj_loadBaleDataFromXML)
    THUtils.setFunctionHook("StoreManager", "loadItem", false, false, self, THDefaultTypes.inj_loadStoreItem)
    THUtils.setFunctionHook("InGameMenuMapFrame", "loadFilters", false, false, self, THDefaultTypes.inj_inGameMenuMapLoadFilters)
    THUtils.setFunctionHook("InGameMenuMapFrame", "generateOverviewOverlay", false, false, self, THDefaultTypes.inj_inGameMenuMapGenerateOverviewOverlay)
    source(self.modPath .. "scripts/placeables/THPlaceableHusbandryMeadow.lua")
end
function THDefaultTypes.onSetMissionInfo(self, mission, missionInfo, missionDynamicInfo, ...)
    THUtils.callSuperClass(self, "onSetMissionInfo", true, mission, missionInfo, missionDynamicInfo, ...)
    self:loadMapData()
    THUtils.setFunctionHook("FSDensityMapUtil", "updateHerbicideArea", false, false, self, THDefaultTypes.inj_densityMapUtilUpdateHerbicideArea)
end
function THDefaultTypes.loadFromMapXML(self, xmlFile, xmlKey, customEnv, baseDirectory, ...)
    local success = THUtils.callSuperClass(self, "loadFromMapXML", true, xmlFile, xmlKey, customEnv, baseDirectory, ...)
    if success then
        local typeConfig = self.typeConfig
        local fruitTypesKey = xmlKey .. ".fruitTypes"
        local fruitTypeConfig = typeConfig.fruitTypes
        if THUtils.hasXMLProperty(xmlFile, fruitTypesKey) then
            local blFruitTypeNames = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, fruitTypesKey, ".blackList")
            if blFruitTypeNames ~= nil and blFruitTypeNames ~= "" then
                local blackListItems = THUtils.splitString(blFruitTypeNames)
                if blackListItems ~= nil and next(blackListItems) ~= nil then
                    for _, fruitTypeName in pairs(blackListItems) do
                        if fruitTypeName ~= "" then
                            fruitTypeConfig.blackList[fruitTypeName:upper()] = true
                        end
                    end
                end
            end
        else
            local blFruitTypesKey = xmlKey .. ".blackList.fruitTypes"
            if THUtils.hasXMLProperty(xmlFile, blFruitTypesKey) then
                xmlFile:iterate(blFruitTypesKey .. ".fruitType", function(_, pFruitTypeKey)
                    local fruitTypeName = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, pFruitTypeKey, "#name")
                    if fruitTypeName ~= nil and fruitTypeName ~= "" then
                        fruitTypeConfig.blackList[fruitTypeName:upper()] = true
                    end
                end)
            end
        end
        local fillTypesKey = xmlKey .. ".fillTypes"
        local fillTypeConfig = typeConfig.fillTypes
        local function addBlackListFillType(pName, pXMLKey)
            if THUtils.argIsValid(type(pName) == THValueType.STRING, "name", pName)
                and THUtils.argIsValid(type(pXMLKey) == THValueType.STRING, "xmlKey", pXMLKey)
            then
                if pName ~= "" then
                    local pId = pName:upper()
                    if not self:getIsRestrictedFillType(pId) then
                        fillTypeConfig.blackList[pId] = true
                    else
                        THUtils.xmlErrorMsg(pXMLKey, nil, THDefaultTypes.MESSAGE.RESTRICTED_FILL_TYPE, pId)
                    end
                end
            end
        end
        if THUtils.hasXMLProperty(xmlFile, fillTypesKey) then
            fillTypeConfig.loadMapFirst = THUtils.getXMLValue(xmlFile, XMLValueType.BOOL, fillTypesKey, "#loadMapFirst", false)
            local blFillTypeNames = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, fillTypesKey, ".blackList")
            if blFillTypeNames ~= nil and blFillTypeNames ~= "" then
                local blackListItems = THUtils.splitString(blFillTypeNames)
                if blackListItems ~= nil and next(blackListItems) ~= nil then
                    for _, fillTypeName in pairs(blackListItems) do
                        addBlackListFillType(fillTypeName, fillTypesKey .. ".blackList")
                    end
                end
            end
        else
            local blFillTypesKey = xmlKey .. ".blackList.fillTypes"
            if THUtils.hasXMLProperty(xmlFile, blFillTypesKey) then
                xmlFile:iterate(blFillTypesKey .. ".fillType", function(_, pFillTypeKey)
                    local fillTypeName = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, pFillTypeKey, "#name")
                    if fillTypeName ~= nil then
                        addBlackListFillType(fillTypeName, pFillTypeKey)
                    end
                end)
            end
        end
        local heightTypesKey = xmlKey .. ".densityMapHeightTypes"
        local heightTypeConfig = typeConfig.heightTypes
        local function addBlackListHeightType(pName, pXMLKey)
            if THUtils.argIsValid(type(pName) == THValueType.STRING, "name", pName)
                and THUtils.argIsValid(type(pXMLKey) == THValueType.STRING, "xmlKey", pXMLKey)
            then
                if pName ~= "" then
                    local pId = pName:upper()
                    if not self:getIsRestrictedHeightType(pId) then
                        heightTypeConfig.blackList[pId] = true
                    else
                        THUtils.xmlErrorMsg(pXMLKey, nil, THDefaultTypes.MESSAGE.RESTRICTED_HEIGHT_TYPE, pId)
                    end
                end
            end
        end
        if THUtils.hasXMLProperty(xmlFile, heightTypesKey) then
            local blHeightTypeNames = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, heightTypesKey, ".blackList")
            if blHeightTypeNames ~= nil and blHeightTypeNames ~= "" then
                local blackListItems = THUtils.splitString(blHeightTypeNames)
                if blackListItems ~= nil and next(blackListItems) ~= nil then
                    for _, heightTypeName in pairs(blackListItems) do
                        addBlackListHeightType(heightTypeName, heightTypesKey .. ".blackList")
                    end
                end
            end
        else
            local blHeightTypesKey = xmlKey .. ".blackList.densityMapHeightTypes"
            if THUtils.hasXMLProperty(xmlFile, blHeightTypesKey) then
                xmlFile:iterate(blHeightTypesKey .. ".densityMapHeightType", function(_, pHeightTypeKey)
                    local heightTypeName = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, pHeightTypeKey, "fillTypeName")
                    if heightTypeName ~= nil then
                        addBlackListHeightType(heightTypeName, pHeightTypeKey)
                    end
                end)
            end
        end
        local balesKey = xmlKey .. ".bales"
        local baleConfig = typeConfig.bales
        local blBalesKey = nil
        if THUtils.hasXMLProperty(xmlFile, balesKey) then
            blBalesKey = balesKey .. ".blackList"
        else
            blBalesKey = xmlKey .. ".blackList.bales"
        end
        if blBalesKey ~= nil
            and THUtils.hasXMLProperty(xmlFile, blBalesKey)
        then
            xmlFile:iterate(blBalesKey .. ".bale", function(_, pBaleKey)
                local baleFilename = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, pBaleKey, "#filename")
                local baleModName = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, pBaleKey, "#modName")
                if baleFilename ~= nil and baleFilename ~= "" then
                    local absFilename = nil
                    if baleModName ~= nil and baleModName ~= "" then
                        local baleModData = self.thModManager:getLoadedMod(baleModName)
                        if baleModData ~= nil then
                            absFilename = THUtils.getFilename(baleFilename, baleModData.path)
                        end
                    else
                        absFilename = THUtils.getFilename(baleFilename)
                    end
                    if absFilename ~= nil and absFilename ~= "" then
                        baleConfig.blackList[absFilename:lower()] = true
                    end
                end
            end)
        end
        local storeItemsKey = xmlKey .. ".storeItems"
        local storeItemConfig = typeConfig.storeItems
        local blStoreItemsKey = nil
        if THUtils.hasXMLProperty(xmlFile, storeItemsKey) then
            blStoreItemsKey = storeItemsKey .. ".blackList"
        else
            blStoreItemsKey = xmlKey .. ".blackList.storeItems"
        end
        if blStoreItemsKey ~= nil
            and THUtils.hasXMLProperty(xmlFile, blStoreItemsKey)
        then
            xmlFile:iterate(blStoreItemsKey .. ".storeItem", function(_, pStoreItemKey)
                local itemFilename = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, pStoreItemKey, "#xmlFilename")
                local itemModName = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, pStoreItemKey, "#modName")
                if itemFilename ~= nil and itemFilename ~= "" then
                    local absFilename = nil
                    if itemModName ~= nil and itemModName ~= "" then
                        local itemModData = self.thModManager:getLoadedMod(itemModName)
                        if itemModData ~= nil then
                            absFilename = THUtils.getFilename(itemFilename, itemModData.path)
                        end
                    else
                        absFilename = THUtils.getFilename(itemFilename)
                    end
                    if absFilename ~= nil and absFilename ~= "" then
                        storeItemConfig.blackList[absFilename:lower()] = true
                    end
                end
            end)
        end
        if THDebugUtil.getIsEnabled(debugFlagId) then
            THUtils.displayMsg("Map type configuration:")
            THDebugUtil.printTable(typeConfig, 4)
        end
        return success
    end
    return false
end
function THDefaultTypes.removeBlackListValues(self, itemList, checkFunc)
    local listType = type(itemList)
    if listType == THValueType.STRING then
        if itemList ~= "" then
            local namesArray = THUtils.splitString(itemList, " ")
            if namesArray ~= nil and next(namesArray) ~= nil then
                local newNames = ""
                for _, otherName in ipairs(namesArray) do
                    if not checkFunc(self, otherName) then
                        if newNames == "" then
                            newNames = otherName
                        else
                            newNames = newNames .. " " .. otherName
                        end
                    end
                end
                return newNames
            end
        end
    elseif listType == THValueType.TABLE then
        if next(itemList) ~= nil then
            local newNameList = {}
            local isArray = false
            if type(itemList[1]) == THValueType.STRING then
                isArray = true
            end
            if isArray then
                for _, otherName in ipairs(itemList) do
                    if not checkFunc(self, otherName) then
                        table.insert(newNameList, otherName)
                    end
                end
            else
                for otherName in pairs(itemList) do
                    if not checkFunc(self, otherName) then
                        newNameList[otherName] = true
                    end
                end
            end
            return newNameList
        end
    elseif itemList ~= nil then
        THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "itemList", itemList)
    end
end
function THDefaultTypes.getIsBlackListFruitType(self, name)
    if THUtils.argIsValid(type(name) == THValueType.STRING, "name", name) then
        local fruitTypeConfig = self.typeConfig.fruitTypes
        local fruitTypeId = string.upper(name)
        if self:getIsBlackListFillType(fruitTypeId)
            or fruitTypeConfig.blackList[fruitTypeId] == true
        then
            return true
        end
    end
    return false
end
function THDefaultTypes.updateFruitTypesXML(self, xmlFile, xmlKey, baseDirectory)
    local xmlFileType = type(xmlFile)
    if THUtils.argIsValid(xmlFileType == THValueType.TABLE or xmlFileType == THValueType.NUMBER, "xmlFile", xmlFile)
        and THUtils.argIsValid(xmlKey == nil or type(xmlKey) == THValueType.STRING, "xmlKey", xmlKey)
        and THUtils.argIsValid(baseDirectory == nil or type(baseDirectory) == THValueType.STRING, "baseDirectory", baseDirectory)
    then
        if xmlFileType == THValueType.NUMBER and xmlFile <= 0 then
            return false
        end
        local xmlRootKey = xmlKey
        if xmlRootKey == nil or xmlRootKey == "" then
            xmlRootKey = THUtils.getXMLRootKey(xmlFile)
        end
        if xmlRootKey ~= nil then
            local xmlIndex = 0
            local success = true
            while true do
                local fruitTypeKey = string.format("%s.fruitTypes.fruitType(%d)", xmlRootKey, xmlIndex)
                local isBlackListFruitType = false
                if not THUtils.hasXMLProperty(xmlFile, fruitTypeKey) then
                    break
                end
                local fruitTypeName = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, fruitTypeKey, "#name")
                local fruitFilename = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, fruitTypeKey, "#filename")
                if fruitTypeName ~= nil and fruitTypeName ~= "" then
                    if self:getIsBlackListFruitType(fruitTypeName) then
                        isBlackListFruitType = true
                        THUtils.displayMsg(THDefaultTypes.MESSAGE.FRUIT_TYPE_REMOVED, fruitTypeName)
                    end
                end
                if not isBlackListFruitType
                    and fruitFilename ~= nil and fruitFilename ~= ""
                then
                    local absFilename = THUtils.getFilename(fruitFilename, baseDirectory)
                    if absFilename ~= nil
                        and THUtils.getFileExists(absFilename)
                    then
                        local foliageXMLFile = THUtils.loadXMLFile(self.dataKey .. "_tempFoliageXML", absFilename)
                        if foliageXMLFile ~= nil then
                            local foliageRootKey = THUtils.getXMLRootKey(foliageXMLFile)
                            if foliageRootKey ~= nil then
                                fruitTypeName = THUtils.getXMLValue(foliageXMLFile, XMLValueType.STRING, foliageRootKey .. ".fruitType", "#name")
                                if fruitTypeName ~= nil and fruitTypeName ~= "" then
                                    if self:getIsBlackListFruitType(fruitTypeName) then
                                        isBlackListFruitType = true
                                        THUtils.displayMsg(THDefaultTypes.MESSAGE.FRUIT_TYPE_REMOVED, fruitTypeName)
                                    end
                                end
                            end
                            THUtils.deleteXMLFile(foliageXMLFile)
                        end
                    end
                end
                if isBlackListFruitType then
                    THUtils.removeXMLProperty(xmlFile, fruitTypeKey)
                else
                    xmlIndex = xmlIndex + 1
                end
            end
            return success
        end
    end
    return false
end
function THDefaultTypes.updateFruitTypeCategoriesXML(self, xmlFile, xmlKey)
    local xmlFileType = type(xmlFile)
    if THUtils.argIsValid(xmlFileType == THValueType.TABLE or xmlFileType == THValueType.NUMBER, "xmlFile", xmlFile)
        and THUtils.argIsValid(xmlKey == nil or type(xmlKey) == THValueType.STRING, "xmlKey", xmlKey)
    then
        if xmlFileType == THValueType.NUMBER and xmlFile <= 0 then
            return false
        end
        local xmlRootKey = xmlKey
        if xmlRootKey == nil or xmlRootKey == "" then
            xmlRootKey = THUtils.getXMLRootKey(xmlFile)
        end
        if xmlRootKey ~= nil then
            local xmlIndex = 0
            local success = true
            while true do
                local categoryKey = string.format("%s.fruitTypeCategories.fruitTypeCategory(%d)", xmlRootKey, xmlIndex)
                local isEntryRemoved = false
                if not THUtils.hasXMLProperty(xmlFile, categoryKey) then
                    break
                end
                local fruitTypeNames = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, categoryKey)
                if fruitTypeNames ~= nil and fruitTypeNames ~= "" then
                    local newFruitTypeNames = self:removeBlackListValues(fruitTypeNames, self.getIsBlackListFruitType)
                    if newFruitTypeNames ~= nil then
                        if newFruitTypeNames == "" then
                            isEntryRemoved = true
                        else
                            THUtils.setXMLValue(xmlFile, XMLValueType.STRING, categoryKey, nil, newFruitTypeNames)
                        end
                    end
                end
                if isEntryRemoved then
                    THUtils.removeXMLProperty(xmlFile, categoryKey)
                else
                    xmlIndex = xmlIndex + 1
                end
            end
            return success
        end
    end
    return false
end
function THDefaultTypes.updateFruitTypeConvertersXML(self, xmlFile, xmlKey)
    local xmlFileType = type(xmlFile)
    if THUtils.argIsValid(xmlFileType == THValueType.TABLE or xmlFileType == THValueType.NUMBER, "xmlFile", xmlFile)
        and THUtils.argIsValid(xmlKey == nil or type(xmlKey) == THValueType.STRING, "xmlKey", xmlKey)
    then
        if xmlFileType == THValueType.NUMBER and xmlFile <= 0 then
            return false
        end
        local xmlRootKey = xmlKey
        if xmlRootKey == nil or xmlRootKey == "" then
            xmlRootKey = THUtils.getXMLRootKey(xmlFile)
        end
        if xmlRootKey ~= nil then
            local xmlIndex = 0
            local success = true
            while true do
                local converterKey = string.format("%s.fruitTypeConverters.fruitTypeConverter(%d)", xmlRootKey, xmlIndex)
                if not THUtils.hasXMLProperty(xmlFile, converterKey) then
                    break
                end
                local conversionXMLIndex = 0
                while true do
                    local conversionKey = string.format("%s.converter(%d)", converterKey, conversionXMLIndex)
                    if not THUtils.hasXMLProperty(xmlFile, conversionKey) then
                        break
                    end
                    local toFruitTypeName = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, conversionKey, "#from")
                    local toFillTypeName = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, conversionKey, "#to")
                    local isConversionRemoved = false
                    if toFruitTypeName ~= nil and toFruitTypeName ~= "" then
                        if self:getIsBlackListFruitType(toFruitTypeName) then
                            isConversionRemoved = true
                        end
                    end
                    if not isConversionRemoved
                        and toFillTypeName ~= nil and toFillTypeName ~= ""
                    then
                        if self:getIsBlackListFillType(toFillTypeName) then
                            isConversionRemoved = true
                        end
                    end
                    if isConversionRemoved then
                        THUtils.removeXMLProperty(xmlFile, conversionKey)
                    else
                        conversionXMLIndex = conversionXMLIndex + 1
                    end
                end
                xmlIndex = xmlIndex + 1
            end
            return success
        end
    end
    return false
end
function THDefaultTypes.updateWeedReplacementsXML(self, xmlFile, xmlKey)
    local xmlFileType = type(xmlFile)
    if THUtils.argIsValid(xmlFileType == THValueType.TABLE or xmlFileType == THValueType.NUMBER, "xmlFile", xmlFile)
        and THUtils.argIsValid(xmlKey == nil or type(xmlKey) == THValueType.STRING, "xmlKey", xmlKey)
    then
        if xmlFileType == THValueType.NUMBER and xmlFile <= 0 then
            return false
        end
        local xmlRootKey = xmlKey
        if xmlRootKey == nil or xmlRootKey == "" then
            xmlRootKey = THUtils.getXMLRootKey(xmlFile)
        end
        if xmlRootKey ~= nil then
            local xmlIndex = 0
            local success = true
            while true do
                local replacementsKey = string.format("%s.replacements(%d)", xmlRootKey, xmlIndex)
                local areReplacementsRemoved = false
                if not THUtils.hasXMLProperty(xmlFile, replacementsKey) then
                    break
                end
                local fruitTypeName = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, replacementsKey, "#fruitType")
                if fruitTypeName ~= nil and fruitTypeName ~= "" then
                    if self:getIsBlackListFruitType(fruitTypeName) then
                        areReplacementsRemoved = true
                    end
                end
                if areReplacementsRemoved then
                    THUtils.removeXMLProperty(xmlFile, replacementsKey)
                    if THDebugUtil.getIsEnabled(debugFlagId) then
                        THUtils.displayMsg("Removed %q from: %s", fruitTypeName, replacementsKey)
                    end
                else
                    xmlIndex = xmlIndex + 1
                end
            end
            return success
        end
    end
    return false
end
function THDefaultTypes.getIsRestrictedFillType(self, name)
    if name ~= nil
        and THUtils.argIsValid(type(name) == THValueType.STRING, "name", name)
    then
        local fillTypeConfig = self.typeConfig.fillTypes
        local fillTypeId = string.upper(name)
        if fillTypeConfig.restricted[fillTypeId] then
            return true
        end
    end
    return false
end
function THDefaultTypes.getIsBlackListFillType(self, name)
    if THUtils.argIsValid(type(name) == THValueType.STRING, "name", name) then
        local fillTypeConfig = self.typeConfig.fillTypes
        local fillTypeId = string.upper(name)
        if not self:getIsRestrictedFillType(fillTypeId) then
            if fillTypeConfig.blackList[fillTypeId] == true then
                return true
            end
        end
    end
    return false
end
function THDefaultTypes.updateFillTypesXML(self, manager, xmlFile, xmlKey, baseDirectory, removeExisting)
    local xmlFileType = type(xmlFile)
    if THUtils.argIsValid(THUtils.getIsType(manager, FillTypeManager), "manager", manager)
        and THUtils.argIsValid(xmlFileType == THValueType.TABLE or xmlFileType == THValueType.NUMBER, "xmlFile", xmlFile)
        and THUtils.argIsValid(xmlKey == nil or type(xmlKey) == THValueType.STRING, "xmlKey", xmlKey)
        and THUtils.argIsValid(baseDirectory == nil or type(baseDirectory) == THValueType.STRING, "baseDirectory", baseDirectory)
        and THUtils.argIsValid(not removeExisting or removeExisting == true, "removeExisting", removeExisting)
    then
        if xmlFileType == THValueType.NUMBER and xmlFile <= 0 then
            return false
        end
        local xmlRootKey = xmlKey
        if xmlRootKey == nil or xmlRootKey == "" then
            xmlRootKey = THUtils.getXMLRootKey(xmlFile)
        end
        if xmlRootKey ~= nil then
            local xmlIndex = 0
            local success = true
            while true do
                local fillTypeKey = string.format("%s.fillTypes.fillType(%d)", xmlRootKey, xmlIndex)
                local isBlackListFillType = false
                if not THUtils.hasXMLProperty(xmlFile, fillTypeKey) then
                    break
                end
                local fillTypeName = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, fillTypeKey, "#name")
                if fillTypeName ~= nil and fillTypeName ~= "" then
                    if self:getIsBlackListFillType(fillTypeName) then
                        isBlackListFillType = true
                        THUtils.displayMsg(THDefaultTypes.MESSAGE.FILL_TYPE_REMOVED, fillTypeName)
                    elseif removeExisting then
                        if manager:getFillTypeByName(fillTypeName) ~= nil then
                            isBlackListFillType = true
                        end
                    end
                end
                if isBlackListFillType then
                    THUtils.removeXMLProperty(xmlFile, fillTypeKey)
                else
                    xmlIndex = xmlIndex + 1
                end
            end
            return success
        end
    end
    return false
end
function THDefaultTypes.updateFillTypeCategoriesXML(self, manager, xmlFile, xmlKey)
    local xmlFileType = type(xmlFile)
    if THUtils.argIsValid(THUtils.getIsType(manager, FillTypeManager), "manager", manager)
        and THUtils.argIsValid(xmlFileType == THValueType.TABLE or xmlFileType == THValueType.NUMBER, "xmlFile", xmlFile)
        and THUtils.argIsValid(xmlKey == nil or type(xmlKey) == THValueType.STRING, "xmlKey", xmlKey)
    then
        if xmlFileType == THValueType.NUMBER and xmlFile <= 0 then
            return false
        end
        local xmlRootKey = xmlKey
        if xmlRootKey == nil or xmlRootKey == "" then
            xmlRootKey = THUtils.getXMLRootKey(xmlFile)
        end
        if xmlRootKey ~= nil then
            local xmlIndex = 0
            local success = true
            while true do
                local categoryKey = string.format("%s.fillTypeCategories.fillTypeCategory(%d)", xmlRootKey, xmlIndex)
                local isEntryRemoved = false
                if not THUtils.hasXMLProperty(xmlFile, categoryKey) then
                    break
                end
                local fillTypeNames = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, categoryKey)
                if fillTypeNames ~= nil and fillTypeNames ~= "" then
                    local newFillTypeNames = self:removeBlackListValues(fillTypeNames, self.getIsBlackListFillType)
                    if newFillTypeNames ~= nil then
                        if newFillTypeNames == "" then
                            isEntryRemoved = true
                        else
                            THUtils.setXMLValue(xmlFile, XMLValueType.STRING, categoryKey, nil, newFillTypeNames)
                        end
                    end
                end
                if isEntryRemoved then
                    THUtils.removeXMLProperty(xmlFile, categoryKey)
                else
                    xmlIndex = xmlIndex + 1
                end
            end
            return success
        end
    end
    return false
end
function THDefaultTypes.updateFillTypeConvertersXML(self, manager, xmlFile, xmlKey)
    local xmlFileType = type(xmlFile)
    if THUtils.argIsValid(THUtils.getIsType(manager, FillTypeManager), "manager", manager)
        and THUtils.argIsValid(xmlFileType == THValueType.TABLE or xmlFileType == THValueType.NUMBER, "xmlFile", xmlFile)
        and THUtils.argIsValid(xmlKey == nil or type(xmlKey) == THValueType.STRING, "xmlKey", xmlKey)
    then
        if xmlFileType == THValueType.NUMBER and xmlFile <= 0 then
            return false
        end
        local xmlRootKey = xmlKey
        if xmlRootKey == nil or xmlRootKey == "" then
            xmlRootKey = THUtils.getXMLRootKey(xmlFile)
        end
        if xmlRootKey ~= nil then
            local xmlIndex = 0
            local success = true
            while true do
                local converterKey = string.format("%s.fillTypeConverters.fillTypeConverter(%d)", xmlRootKey, xmlIndex)
                if not THUtils.hasXMLProperty(xmlFile, converterKey) then
                    break
                end
                local conversionXMLIndex = 0
                while true do
                    local conversionKey = string.format("%s.converter(%d)", converterKey, conversionXMLIndex)
                    if not THUtils.hasXMLProperty(xmlFile, conversionKey) then
                        break
                    end
                    local fromFillTypeName = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, conversionKey, "#from")
                    local toFillTypeName = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, conversionKey, "#to")
                    local isConversionRemoved = false
                    if fromFillTypeName ~= nil and fromFillTypeName ~= "" then
                        if self:getIsBlackListFillType(fromFillTypeName) then
                            isConversionRemoved = true
                        end
                    end
                    if not isConversionRemoved
                        and toFillTypeName ~= nil and toFillTypeName ~= ""
                    then
                        if self:getIsBlackListFillType(toFillTypeName) then
                            isConversionRemoved = true
                        end
                    end
                    if isConversionRemoved then
                        THUtils.removeXMLProperty(xmlFile, conversionKey)
                    else
                        conversionXMLIndex = conversionXMLIndex + 1
                    end
                end
                xmlIndex = xmlIndex + 1
            end
            return success
        end
    end
    return false
end
function THDefaultTypes.updateFillTypeSoundsXML(self, manager, xmlFile, xmlKey)
    local xmlFileType = type(xmlFile)
    if THUtils.argIsValid(THUtils.getIsType(manager, FillTypeManager), "manager", manager)
        and THUtils.argIsValid(xmlFileType == THValueType.TABLE or xmlFileType == THValueType.NUMBER, "xmlFile", xmlFile)
        and THUtils.argIsValid(xmlKey == nil or type(xmlKey) == THValueType.STRING, "xmlKey", xmlKey)
    then
        if xmlFileType == THValueType.NUMBER and xmlFile <= 0 then
            return false
        end
        local xmlRootKey = xmlKey
        if xmlRootKey == nil or xmlRootKey == "" then
            xmlRootKey = THUtils.getXMLRootKey(xmlFile)
        end
        if xmlRootKey ~= nil then
            local xmlIndex = 0
            local success = true
            while true do
                local soundKey = string.format("%s.fillTypeSounds.fillTypeSound(%d)", xmlRootKey, xmlIndex)
                if not THUtils.hasXMLProperty(xmlFile, soundKey) then
                    break
                end
                local fillTypeNames = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, soundKey, "#fillTypes")
                if fillTypeNames ~= nil and fillTypeNames ~= "" then
                    local newFillTypeNames = self:removeBlackListValues(fillTypeNames, self.getIsBlackListFillType)
                    if newFillTypeNames ~= nil then
                        if newFillTypeNames == "" then
                            THUtils.removeXMLProperty(xmlFile, soundKey .. "#fillTypes")
                        else
                            THUtils.setXMLValue(xmlFile, XMLValueType.STRING, soundKey, "#fillTypes", newFillTypeNames)
                        end
                    end
                end
                xmlIndex = xmlIndex + 1
            end
            return success
        end
    end
    return false
end
function THDefaultTypes.getIsRestrictedHeightType(self, name)
    if name ~= nil and name ~= ""
        and THUtils.argIsValid(type(name) == THValueType.STRING, "name", name)
    then
        local heightTypeConfig = self.typeConfig.heightTypes
        local heightTypeId = string.upper(name)
        if heightTypeConfig.restricted[heightTypeId] then
            return true
        end
    end
    return false
end
function THDefaultTypes.getIsBlackListHeightType(self, name)
    if THUtils.argIsValid(type(name) == THValueType.STRING, "name", name) then
        local heightTypeConfig = self.typeConfig.heightTypes
        local heightTypeId = string.upper(name)
        if not self:getIsRestrictedHeightType(heightTypeId) then
            if self:getIsBlackListFillType(heightTypeId)
                or heightTypeConfig.blackList[heightTypeId] == true
            then
                return true
            end
        end
    end
    return false
end
function THDefaultTypes.updateHeightTypesXML(self, xmlFile, xmlKey, baseDirectory)
    local xmlFileType = type(xmlFile)
    if THUtils.argIsValid(xmlFileType == THValueType.TABLE or xmlFileType == THValueType.NUMBER, "xmlFile", xmlFile)
        and THUtils.argIsValid(xmlKey == nil or type(xmlKey) == THValueType.STRING, "xmlKey", xmlKey)
        and THUtils.argIsValid(baseDirectory == nil or type(baseDirectory) == THValueType.STRING, "baseDirectory", baseDirectory)
    then
        if xmlFileType == THValueType.NUMBER and xmlFile <= 0 then
            return false
        end
        local xmlRootKey = xmlKey
        if xmlRootKey == nil or xmlRootKey == "" then
            xmlRootKey = THUtils.getXMLRootKey(xmlFile)
        end
        if xmlRootKey ~= nil then
            local xmlIndex = 0
            local success = true
            while true do
                local heightTypeKey = string.format("%s.densityMapHeightTypes.densityMapHeightType(%d)", xmlRootKey, xmlIndex)
                if not THUtils.hasXMLProperty(xmlFile, heightTypeKey) then
                    break
                end
                local fillTypeName = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, heightTypeKey, "#fillTypeName")
                local isBlackListHeightType = false
                if fillTypeName ~= nil and fillTypeName ~= "" then
                    if self:getIsBlackListHeightType(fillTypeName) then
                        isBlackListHeightType = true
                        THUtils.displayMsg(THDefaultTypes.MESSAGE.HEIGHT_TYPE_REMOVED, fillTypeName)
                    end
                end
                if isBlackListHeightType then
                    THUtils.removeXMLProperty(xmlFile, heightTypeKey)
                else
                    xmlIndex = xmlIndex + 1
                end
            end
            return success
        end
    end
    return false
end
function THDefaultTypes.getIsBlackListBale(self, filename, baseDirectory, isFullCheck)
    if THUtils.argIsValid(type(filename) == THValueType.STRING, "filename", filename)
        and THUtils.argIsValid(baseDirectory == nil or type(baseDirectory) == THValueType.STRING, "baseDirectory", baseDirectory)
        and THUtils.argIsValid(not isFullCheck or isFullCheck == true, "isFullCheck", isFullCheck)
    then
        local absFilename = nil
        if baseDirectory ~= nil and baseDirectory ~= "" then
            absFilename = THUtils.getFilename(filename, baseDirectory)
        else
            absFilename = THUtils.getFilename(filename)
        end
        if absFilename ~= nil and absFilename ~= "" then
            local baleConfig = self.typeConfig.bales
            local isBlackListBale = baleConfig.blackList[absFilename:lower()]
            if not isBlackListBale and isFullCheck == true
                and THUtils.getFileExists(absFilename)
            then
                local xmlFile = THUtils.loadXMLFile(self.dataKey .. "_tempBaleXML", absFilename)
                if xmlFile ~= nil then
                    local xmlRootKey = THUtils.getXMLRootKey(xmlFile)
                    if xmlRootKey ~= nil then
                        local updateSuccess, needsRemoval = self:updateBaleDataXML(xmlFile, xmlRootKey, baseDirectory)
                        if updateSuccess and needsRemoval then
                            isBlackListBale = true
                        end
                    end
                    THUtils.deleteXMLFile(xmlFile)
                end
            end
            return isBlackListBale == true, absFilename
        end
    end
    return false
end
function THDefaultTypes.getIsBlackListBaleFromXML(self, xmlFile, xmlKey, baseDirectory, isFullCheck)
    local xmlFileType = type(xmlFile)
    if THUtils.argIsValid(xmlFileType == THValueType.TABLE or xmlFileType == THValueType.NUMBER, "xmlFile", xmlFile)
        and THUtils.argIsValid(type(xmlKey) == THValueType.STRING, "xmlKey", xmlKey)
    then
        if xmlFileType == THValueType.NUMBER and xmlFile <= 0 then
            return false
        end
        local filename = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, xmlKey, "#filename")
        if filename ~= nil and filename ~= "" then
            return self:getIsBlackListBale(filename, baseDirectory, isFullCheck), filename
        end
    end
    return false
end
function THDefaultTypes.updateBaleDataXML(self, xmlFile, xmlKey, baseDirectory)
    local xmlFileType = type(xmlFile)
    if THUtils.argIsValid(xmlFileType == THValueType.TABLE or xmlFileType == THValueType.NUMBER, "xmlFile", xmlFile)
        and THUtils.argIsValid(xmlKey == nil or type(xmlKey) == THValueType.STRING, "xmlKey", xmlKey)
        and THUtils.argIsValid(baseDirectory == nil or type(baseDirectory) == THValueType.STRING, "baseDirectory", baseDirectory)
    then
        if xmlFileType == THValueType.NUMBER and xmlFile <= 0 then
            return false
        end
        local xmlRootKey = xmlKey
        if xmlRootKey == nil or xmlRootKey == "" then
            xmlRootKey = THUtils.getXMLRootKey(xmlFile)
        end
        if xmlRootKey ~= nil then
            local fillTypeXMLIndex = 0
            local hasFillTypeEntry = false
            local success, isBlackListBale = true, false
            while true do
                local fillTypeKey = string.format("%s.fillTypes.fillType(%d)", xmlRootKey, fillTypeXMLIndex)
                local isBlackListFillType = false
                if not THUtils.hasXMLProperty(xmlFile, fillTypeKey) then
                    break
                end
                local fillTypeName = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, fillTypeKey, "#name")
                if fillTypeName ~= nil and fillTypeName ~= "" then
                    if self:getIsBlackListFillType(fillTypeName) then
                        isBlackListFillType = true
                    end
                end
                if isBlackListFillType then
                    THUtils.removeXMLProperty(xmlFile, fillTypeKey)
                else
                    fillTypeXMLIndex = fillTypeXMLIndex + 1
                end
                hasFillTypeEntry = true
            end
            if success then
                if hasFillTypeEntry and fillTypeXMLIndex == 0 then
                    isBlackListBale = true
                end
                return success, isBlackListBale
            end
        end
    end
    return false
end
function THDefaultTypes.getIsBlackListStoreItem(self, filename, baseDirectory)
    if THUtils.argIsValid(type(filename) == THValueType.STRING, "filename", filename)
        and THUtils.argIsValid(baseDirectory == nil or type(baseDirectory) == THValueType.STRING, "baseDirectory", baseDirectory)
    then
        local absFilename = nil
        if baseDirectory ~= nil and baseDirectory ~= "" then
            absFilename = THUtils.getFilename(filename, baseDirectory)
        else
            absFilename = THUtils.getFilename(filename)
        end
        if absFilename ~= nil and absFilename ~= "" then
            local storeItemConfig = self.typeConfig.storeItems
            local isBlackListStoreItem = storeItemConfig.blackList[absFilename:lower()]
            if isBlackListStoreItem == true then
                return true, absFilename
            end
        end
    end
    return false
end
function THDefaultTypes.updateStoreItemsXML(self, xmlFile, xmlKey, baseDirectory)
    local xmlFileType = type(xmlFile)
    if THUtils.argIsValid(xmlFileType == THValueType.TABLE or xmlFileType == THValueType.NUMBER, "xmlFile", xmlFile)
        and THUtils.argIsValid(xmlKey == nil or type(xmlKey) == THValueType.STRING, "xmlKey", xmlKey)
        and THUtils.argIsValid(baseDirectory == nil or type(baseDirectory) == THValueType.STRING, "baseDirectory", baseDirectory)
    then
        if xmlFileType == THValueType.NUMBER and xmlFile <= 0 then
            return false
        end
        local xmlRootKey = xmlKey
        if xmlRootKey == nil or xmlRootKey == "" then
            xmlRootKey = THUtils.getXMLRootKey(xmlFile)
        end
        if xmlRootKey ~= nil then
            local storeItemXMLIndex = 0
            local success = true
            while true do
                local storeItemKey = string.format("%s.storeItem(%d)", xmlRootKey, storeItemXMLIndex)
                local isBlackListItem = false
                if not THUtils.hasXMLProperty(xmlFile, storeItemKey) then
                    break
                end
                local itemFilename = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, storeItemKey, "#xmlFilename")
                if itemFilename ~= nil and itemFilename ~= "" then
                    local isItemFound, absItemFilename = self:getIsBlackListStoreItem(itemFilename, baseDirectory)
                    if isItemFound then
                        isBlackListItem = true
                        THUtils.displayMsg(THDefaultTypes.MESSAGE.STORE_ITEM_REMOVED, absItemFilename)
                    end
                end
                if isBlackListItem then
                    THUtils.removeXMLProperty(xmlFile, storeItemKey)
                else
                    storeItemXMLIndex = storeItemXMLIndex + 1
                end
            end
            return success
        end
    end
    return false
end
function THDefaultTypes.updateMissionVehiclesXML(self, xmlFile, xmlKey, baseDirectory)
    local xmlFileType = type(xmlFile)
    if THUtils.argIsValid(xmlFileType == THValueType.TABLE or xmlFileType == THValueType.NUMBER, "xmlFile", xmlFile)
        and THUtils.argIsValid(xmlKey == nil or type(xmlKey) == THValueType.STRING, "xmlKey", xmlKey)
        and THUtils.argIsValid(baseDirectory == nil or type(baseDirectory) == THValueType.STRING, "baseDirectory", baseDirectory)
    then
        if xmlFileType == THValueType.NUMBER and xmlFile <= 0 then
            return false
        end
        local xmlRootKey = xmlKey
        if xmlRootKey == nil or xmlRootKey == "" then
            xmlRootKey = THUtils.getXMLRootKey(xmlFile)
        end
        if xmlRootKey ~= nil then
            local missionXMLIndex = 0
            local success = true
            while true do
                local missionKey = string.format("%s.mission(%d)", xmlRootKey, missionXMLIndex)
                if not THUtils.hasXMLProperty(xmlFile, missionKey) then
                    break
                end
                local groupXMLIndex = 0
                while true do
                    local groupKey = string.format("%s.group(%d)", missionKey, groupXMLIndex)
                    local isGroupRemoved = false
                    if not THUtils.hasXMLProperty(xmlFile, groupKey) then
                        break
                    end
                    local vehicleXMLIndex = 0
                    while true do
                        local vehicleKey = string.format("%s.vehicle(%d)", groupKey, vehicleXMLIndex)
                        if not THUtils.hasXMLProperty(xmlFile, vehicleKey) then
                            break
                        end
                        local storeItemFilename = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, vehicleKey, "#filename")
                        if storeItemFilename ~= nil and storeItemFilename ~= "" then
                            if self:getIsBlackListStoreItem(storeItemFilename, baseDirectory) then
                                isGroupRemoved = true
                                break
                            end
                        end
                        vehicleXMLIndex = vehicleXMLIndex + 1
                    end
                    if isGroupRemoved then
                        THUtils.removeXMLProperty(xmlFile, groupKey)
                    else
                        groupXMLIndex = groupXMLIndex + 1
                    end
                end
                missionXMLIndex = missionXMLIndex + 1
            end
            return success
        end
    end
    return false
end
function THDefaultTypes.inj_loadXMLFile(self, superFunc, xmlName, xmlFilename, ...)
    local function appendFunc(rXMLFile, ...)
        if type(xmlName) == THValueType.STRING
            and type(xmlFilename) == THValueType.STRING
            and rXMLFile ~= nil and rXMLFile ~= 0
        then
            THUtils.call(function()
                local xmlId = string.upper(xmlName)
                local baseDirectory = nil
                if xmlFilename ~= "" then
                    local modData = self.thModManager:getLoadedModByFilename(xmlFilename)
                    if modData ~= nil then
                        baseDirectory = modData.path
                    end
                end
                local xmlRootKey = THUtils.getXMLRootKey(rXMLFile)
                if xmlRootKey ~= nil and xmlRootKey ~= "" then
                    if xmlId == "STOREITEMSXML" then
                        self:updateStoreItemsXML(rXMLFile, xmlRootKey, baseDirectory)
                    elseif xmlId == "MISSIONVEHICLES" then
                        self:updateMissionVehiclesXML(rXMLFile, xmlRootKey, baseDirectory)
                    end
                end
            end)
        end
        return rXMLFile, ...
    end
    return appendFunc(superFunc(xmlName, xmlFilename, ...))
end
function THDefaultTypes.inj_loadDataFromMapXML(self, superFunc, xmlFile, xmlDataKey, ...)
    local function appendFunc(...)
        THUtils.call(function()
            if type(xmlDataKey) == THValueType.STRING then
                local xmlDataId = string.upper(xmlDataKey)
                if xmlDataId == "FILLTYPES" then
                    local fillTypeManager = self.typeManagers.fillTypes
                    local fillTypeConfig = self.typeConfig.fillTypes
                    if not fillTypeConfig.isMapLoadFinished then
                        fillTypeConfig.isMapLoadFinished = true
                        if fillTypeConfig.loadMapFirst and not fillTypeConfig.isDefaultTypesLoadFinished then
                            fillTypeManager:loadDefaultTypes()
                        end
                    end
                end
            end
        end)
        return ...
    end
    return appendFunc(superFunc(xmlFile, xmlDataKey, ...))
end
function THDefaultTypes.inj_loadFruitTypeMapData(self, superFunc, manager, ...)
    if manager ~= nil then
        THUtils.call(function()
            if manager ~= self.typeManagers.fruitTypes then
                self.typeManagers.fruitTypes = manager
            end
        end)
    end
    return superFunc(manager, ...)
end
function THDefaultTypes.inj_loadFruitTypes(self, superFunc, manager, xmlFile, param2, baseDirectory, ...)
    local xmlFileType = type(xmlFile)
    if (xmlFileType == THValueType.TABLE or (xmlFileType == THValueType.NUMBER and xmlFile > 0))
        and (baseDirectory == nil or type(baseDirectory) == THValueType.STRING)
    then
        THUtils.call(function()
            local xmlRootKey = THUtils.getXMLRootKey(xmlFile)
            if xmlRootKey ~= nil and xmlRootKey ~= "" then
                self:updateFruitTypesXML(xmlFile, xmlRootKey, baseDirectory)
            end
        end)
    end
    return superFunc(manager, xmlFile, param2, baseDirectory, ...)
end
function THDefaultTypes.inj_loadMapFruitTypes(self, superFunc, manager, xmlFile, param2, baseDirectory, ...)
    local xmlFileType = type(xmlFile)
    if (xmlFileType == THValueType.TABLE or (xmlFileType == THValueType.NUMBER and xmlFile > 0))
        and (baseDirectory == nil or type(baseDirectory) == THValueType.STRING)
    then
        THUtils.call(function()
            local xmlRootKey = THUtils.getXMLRootKey(xmlFile)
            if xmlRootKey ~= nil and xmlRootKey ~= "" then
                self:updateFruitTypesXML(xmlFile, xmlRootKey, baseDirectory)
            end
        end)
    end
    return superFunc(manager, xmlFile, param2, baseDirectory, ...)
end
function THDefaultTypes.inj_loadFruitTypeCategoriesFromXML(self, superFunc, manager, xmlFile, xmlKey, ...)
    local xmlFileType = type(xmlFile)
    if (xmlKey == nil or type(xmlKey) == THValueType.STRING)
        and (xmlFileType == THValueType.TABLE or (xmlFileType == THValueType.NUMBER and xmlFile > 0))
    then
        THUtils.call(function()
            self:updateFruitTypeCategoriesXML(xmlFile, xmlKey)
        end)
    end
    return superFunc(manager, xmlFile, xmlKey, ...)
end
function THDefaultTypes.inj_loadFruitTypeConvertersFromXML(self, superFunc, manager, xmlFile, xmlKey, ...)
    local xmlFileType = type(xmlFile)
    if (xmlKey == nil or type(xmlKey) == THValueType.STRING)
        and (xmlFileType == THValueType.TABLE or (xmlFileType == THValueType.NUMBER and xmlFile > 0))
    then
        THUtils.call(function()
            self:updateFruitTypeConvertersXML(xmlFile, xmlKey)
        end)
    end
    return superFunc(manager, xmlFile, xmlKey, ...)
end
function THDefaultTypes.inj_loadFruitTypeFromFoliageXMLFile(self, superFunc, desc, xmlFilename, ...)
    local isBlackListFruitType = false
    local isFruitTypeSwapped = false
    if xmlFilename ~= nil then
        THUtils.call(function()
            local fruitTypeManager = self.typeManagers.fruitTypes
            if THUtils.getFileExists(xmlFilename) then
                local xmlFile = THUtils.loadXMLFile(self.dataKey .. "_tempFoliageXML", xmlFilename)
                if xmlFile ~= nil then
                    local fruitTypeName = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, "foliageType.fruitType", "#name")
                    if fruitTypeName ~= nil and fruitTypeName ~= "" then
                        if self:getIsBlackListFruitType(fruitTypeName) then
                            isBlackListFruitType = true
                            THUtils.displayMsg(THDefaultTypes.MESSAGE.FRUIT_TYPE_SKIPPED, fruitTypeName)
                        else
                            local otherDesc = fruitTypeManager:getFruitTypeByName(fruitTypeName)
                            if otherDesc ~= nil and otherDesc ~= desc then
                                THUtils.errorMsg(false, THDefaultTypes.MESSAGE.DUPLICATE_FRUIT_TYPE, fruitTypeName)
                                isFruitTypeSwapped = true
                                desc = otherDesc
                            end
                        end
                    end
                    THUtils.deleteXMLFile(xmlFile)
                end
            end
        end)
    end
    local function appendFunc(rSuccess, ...)
        if rSuccess and isFruitTypeSwapped then
            rSuccess = false
        end
        return rSuccess, ...
    end
    if not isBlackListFruitType then
        return appendFunc(superFunc(desc, xmlFilename, ...))
    end
    return false
end
function THDefaultTypes.inj_loadWeedSystemReplacements(self, superFunc, weedSystem, xmlFile, xmlKey, infoTable, ...)
    local xmlFileType = type(xmlFile)
    if (xmlKey == nil or type(xmlKey) == THValueType.STRING)
        and (xmlFileType == THValueType.TABLE or (xmlFileType == THValueType.NUMBER and xmlFile > 0))
    then
        THUtils.call(function()
            self:updateWeedReplacementsXML(xmlFile, xmlKey)
        end)
    end
    local function appendFunc(...)
        if infoTable ~= nil then
            THUtils.call(function()
                if infoTable.custom == nil then
                    self.needsUpdateHerbicideAreaFix = true
                end
            end)
        end
        return ...
    end
    return appendFunc(superFunc(weedSystem, xmlFile, xmlKey, infoTable, ...))
end
function THDefaultTypes.inj_loadFillTypeMapData(self, superFunc, manager, ...)
    if manager ~= nil then
        THUtils.call(function()
            if manager ~= self.typeManagers.fillTypes then
                self.typeManagers.fillTypes = manager
            end
        end)
    end
    return superFunc(manager, ...)
end
function THDefaultTypes.inj_loadDefaultFillTypes(self, superFunc, manager, ...)
    local allowRun = true
    THUtils.call(function()
        local fillTypeConfig = self.typeConfig.fillTypes
        if fillTypeConfig.loadMapFirst then
            if not fillTypeConfig.isMapLoadFinished then
                allowRun = false
            end
        end
        if allowRun then
            fillTypeConfig.isDefaultTypesLoading = true
        end
    end)
    local function appendFunc(...)
        THUtils.call(function()
            local fillTypeConfig = self.typeConfig.fillTypes
            fillTypeConfig.isDefaultTypesLoading = false
            fillTypeConfig.isDefaultTypesLoadFinished = true
        end)
        return ...
    end
    if allowRun then
        return appendFunc(superFunc(manager, ...))
    end
end
function THDefaultTypes.inj_loadFillTypes(self, superFunc, manager, xmlFile, baseDirectory, isBaseType, ...)
    local xmlFileType = type(xmlFile)
    if (xmlFileType == THValueType.TABLE or (xmlFileType == THValueType.NUMBER and xmlFile > 0))
        and (baseDirectory == nil or type(baseDirectory) == THValueType.STRING)
        and (not isBaseType or isBaseType == true)
    then
        THUtils.call(function()
            local xmlRootKey = THUtils.getXMLRootKey(xmlFile)
            if xmlRootKey ~= nil then
                local fillTypeConfig = self.typeConfig.fillTypes
                local removeExisting = isBaseType == true
                self:updateFillTypesXML(manager, xmlFile, xmlRootKey, baseDirectory, removeExisting)
                self:updateFillTypeCategoriesXML(manager, xmlFile, xmlRootKey)
                self:updateFillTypeConvertersXML(manager, xmlFile, xmlRootKey)
                self:updateFillTypeSoundsXML(manager, xmlFile, xmlRootKey)
                if fillTypeConfig.loadMapFirst then
                    if isBaseType then
                        if fillTypeConfig.isDefaultTypesLoading then
                            isBaseType = false
                        end
                    else
                        if not fillTypeConfig.areBaseTypesLoaded then
                            isBaseType = true
                        end
                    end
                end
            end
        end)
    end
    local function appendFunc(rSuccess, ...)
        THUtils.call(function()
            local fillTypeConfig = self.typeConfig.fillTypes
            if isBaseType then
                if rSuccess or fillTypeConfig.isDefaultTypesLoading then
                    fillTypeConfig.areBaseTypesLoaded = true
                end
            end
        end)
        return rSuccess, ...
    end
    return appendFunc(superFunc(manager, xmlFile, baseDirectory, isBaseType, ...))
end
function THDefaultTypes.inj_loadFillTypeFromXMLFile(self, superFunc, desc, xmlFile, xmlKey, ...)
    local xmlFileType = type(xmlFile)
    local isBlackListFillType = false
    local isFillTypeSwapped = false
    if type(xmlKey) == THValueType.STRING
        and (xmlFileType == THValueType.TABLE or (xmlFileType == THValueType.NUMBER and xmlFile > 0))
    then
        THUtils.call(function()
            local fillTypeManager = self.typeManagers.fillTypes
            local fillTypeName = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, xmlKey, "#name")
            if fillTypeName ~= nil and fillTypeName ~= "" then
                if self:getIsBlackListFillType(fillTypeName) then
                    isBlackListFillType = true
                    THUtils.displayMsg(THDefaultTypes.MESSAGE.FILL_TYPE_SKIPPED, fillTypeName)
                else
                    local otherDesc = fillTypeManager:getFillTypeByName(fillTypeName)
                    if otherDesc ~= nil and otherDesc ~= desc then
                        THUtils.errorMsg(false, THDefaultTypes.MESSAGE.DUPLICATE_FILL_TYPE, fillTypeName)
                        isFillTypeSwapped = true
                        desc = otherDesc
                    end
                end
            end
        end)
    end
    local function appendFunc(rSuccess, ...)
        if rSuccess and isFillTypeSwapped then
            rSuccess = false
        end
        return rSuccess, ...
    end
    if not isBlackListFillType then
        return appendFunc(superFunc(desc, xmlFile, xmlKey, ...))
    end
    return false
end
function THDefaultTypes.inj_loadAnimalFoodRecipes(self, superFunc, animalFoodSystem, xmlFile, ...)
    local xmlFileType = type(xmlFile)
    if xmlFileType == THValueType.TABLE or (xmlFileType == THValueType.NUMBER and xmlFile > 0) then
        THUtils.call(function()
            local recipeXMLIndex = 0
            while true do
                local recipeKey = string.format("animalFood.recipes.recipe(%d)", recipeXMLIndex)
                local isRecipeRemoved = false
                if not THUtils.hasXMLProperty(xmlFile, recipeKey) then
                    break
                end
                local fillTypeName = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, recipeKey, "#fillType")
                if fillTypeName ~= nil and fillTypeName ~= "" then
                    if self:getIsBlackListFillType(fillTypeName) then
                        isRecipeRemoved = true
                    end
                end
                if isRecipeRemoved then
                    THUtils.removeXMLProperty(xmlFile, recipeKey)
                else
                    recipeXMLIndex = recipeXMLIndex + 1
                end
            end
        end)
    end
    return superFunc(animalFoodSystem, xmlFile, ...)
end
function THDefaultTypes.inj_loadAnimalFoodMixtures(self, superFunc, animalFoodSystem, xmlFile, ...)
    local xmlFileType = type(xmlFile)
    if xmlFileType == THValueType.TABLE or (xmlFileType == THValueType.NUMBER and xmlFile > 0) then
        THUtils.call(function()
            local mixtureXMLIndex = 0
            while true do
                local mixtureKey = string.format("animalFood.mixtures.mixture(%d)", mixtureXMLIndex)
                local isMixtureRemoved = false
                if not THUtils.hasXMLProperty(xmlFile, mixtureKey) then
                    break
                end
                local fillTypeName = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, mixtureKey, "#fillType")
                if fillTypeName ~= nil and fillTypeName ~= "" then
                    if self:getIsBlackListFillType(fillTypeName) then
                        isMixtureRemoved = true
                    end
                end
                if isMixtureRemoved then
                    THUtils.removeXMLProperty(xmlFile, mixtureKey)
                else
                    mixtureXMLIndex = mixtureXMLIndex + 1
                end
            end
        end)
    end
    return superFunc(animalFoodSystem, xmlFile, ...)
end
function THDefaultTypes.inj_getAnimalFoodFillTypesFromXML(self, superFunc, animalFoodSystem, valuesTable, valuesTable2, xmlFile, xmlKey, ...)
    local hasFillTypeNames = true
    if xmlFile ~= nil and xmlKey ~= nil then
        THUtils.call(function()
            local fillTypeNames = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, xmlKey)
            if fillTypeNames ~= nil and fillTypeNames ~= "" then
                local newFillTypeNames = self:removeBlackListValues(fillTypeNames, self.getIsBlackListFillType)
                if newFillTypeNames ~= nil then
                    if newFillTypeNames == "" then
                        hasFillTypeNames = false
                    else
                        THUtils.setXMLValue(xmlFile, XMLValueType.STRING, xmlKey, nil, newFillTypeNames)
                    end
                end
            end
        end)
    end
    if hasFillTypeNames then
        return superFunc(animalFoodSystem, valuesTable, valuesTable2, xmlFile, xmlKey, ...)
    end
    return false
end
function THDefaultTypes.inj_loadDensityMapHeightTypes(self, superFunc, manager, xmlFile, ...)
    local xmlFileType = type(xmlFile)
    if xmlFileType == THValueType.TABLE or (xmlFileType == THValueType.NUMBER and xmlFile > 0) then
        THUtils.call(function()
            local xmlRootKey = THUtils.getXMLRootKey(xmlFile)
            if xmlRootKey ~= nil and xmlRootKey ~= "" then
                self:updateHeightTypesXML(xmlFile, xmlRootKey)
            end
        end)
    end
    return superFunc(manager, xmlFile, ...)
end
function THDefaultTypes.inj_loadBaleFromXML(self, superFunc, manager, xmlFile, xmlKey, baseDirectory, ...)
    local xmlFileType = type(xmlFile)
    local isBlackListBale = false
    if (xmlKey == nil or type(xmlKey) == THValueType.STRING)
        and (xmlFileType == THValueType.TABLE or (xmlFileType == THValueType.NUMBER and xmlFile > 0))
    then
        THUtils.call(function()
            local isBaleFound, absBaleFilename = self:getIsBlackListBaleFromXML(xmlFile, xmlKey, baseDirectory, true)
            if isBaleFound then
                isBlackListBale = true
                THUtils.displayMsg(THDefaultTypes.MESSAGE.BALE_REMOVED, absBaleFilename)
            end
        end)
    end
    if not isBlackListBale then
        return superFunc(manager, xmlFile, xmlKey, baseDirectory, ...)
    end
    return false
end
function THDefaultTypes.inj_loadModBaleFromXML(self, superFunc, manager, xmlFile, xmlKey, baseDirectory, ...)
    local xmlFileType = type(xmlFile)
    local isBlackListBale = false
    if (xmlKey == nil or type(xmlKey) == THValueType.STRING)
        and (xmlFileType == THValueType.TABLE or (xmlFileType == THValueType.NUMBER and xmlFile > 0))
    then
        THUtils.call(function()
            local isBaleFound, absBaleFilename = self:getIsBlackListBaleFromXML(xmlFile, xmlKey, baseDirectory, true)
            if isBaleFound then
                isBlackListBale = true
                THUtils.displayMsg(THDefaultTypes.MESSAGE.BALE_REMOVED, absBaleFilename)
            end
        end)
    end
    if not isBlackListBale then
        return superFunc(manager, xmlFile, xmlKey, baseDirectory, ...)
    end
    return false
end
function THDefaultTypes.inj_loadBaleDataFromXML(self, superFunc, manager, loadInfo, xmlFile, baseDirectory, ...)
    local xmlFileType = type(xmlFile)
    local isBlackListBale = false
    if xmlFileType == THValueType.TABLE or (xmlFileType == THValueType.NUMBER and xmlFile > 0) then
        THUtils.call(function()
            local absBaleFilename = nil
            if type(loadInfo) == THValueType.TABLE
                and type(loadInfo.xmlFilename) == THValueType.STRING and loadInfo.xmlFilename ~= ""
            then
                absBaleFilename = loadInfo.xmlFilename
                if self:getIsBlackListBale(absBaleFilename) then
                    isBlackListBale = true
                    THUtils.displayMsg(THDefaultTypes.MESSAGE.BALE_SKIPPED, absBaleFilename)
                end
            end
            if not isBlackListBale then
                local xmlRootKey = THUtils.getXMLRootKey(xmlFile)
                if xmlRootKey ~= nil then
                    local updateSuccess, needsRemoval = self:updateBaleDataXML(xmlFile, xmlRootKey, baseDirectory)
                    if updateSuccess and needsRemoval then
                        isBlackListBale = true
                        if absBaleFilename ~= nil then
                            THUtils.displayMsg(THDefaultTypes.MESSAGE.BALE_SKIPPED, absBaleFilename)
                        end
                    end
                end
            end
        end)
    end
    if not isBlackListBale then
        return superFunc(manager, loadInfo, xmlFile, baseDirectory, ...)
    end
    return false
end
function THDefaultTypes.inj_loadStoreItem(self, superFunc, manager, filename, baseDirectory, ...)
    local isBlackListItem = false
    if filename ~= nil and filename ~= "" then
        THUtils.call(function()
            local isItemFound, absItemFilename = self:getIsBlackListStoreItem(filename, baseDirectory)
            if isItemFound then
                isBlackListItem = true
                THUtils.displayMsg(THDefaultTypes.MESSAGE.STORE_ITEM_SKIPPED, absItemFilename)
            end
        end)
    end
    if not isBlackListItem then
        return superFunc(manager, filename, baseDirectory, ...)
    end
end
local function updateInGameMenuMapFruitTypeFilters(self, frame)
    if THUtils.argIsValid(THUtils.getIsType(self, THDefaultTypes), "self", self)
        and THUtils.argIsValid(THUtils.getIsType(frame, InGameMenuMapFrame), "frame", frame)
    then
        local customData = THUtils.getDataTable(frame, self.dataKey)
        local displayCropTypes = frame.displayCropTypes
        local fruitTypeFilter = frame.fruitTypeFilter
        if customData ~= nil
            and displayCropTypes ~= nil and fruitTypeFilter ~= nil
        then
            local fruitTypeFilterStates = customData.fruitTypeFilterStates
            for dataIndex, cropTypeData in pairs(displayCropTypes) do
                local fruitTypeIndex = cropTypeData.fruitTypeIndex
                local isVisible = fruitTypeFilterStates[dataIndex] == true
                if fruitTypeIndex ~= nil and fruitTypeIndex ~= FruitType.UNKNOWN then
                    fruitTypeFilter[fruitTypeIndex] = isVisible
                end
            end
        end
    end
end
function THDefaultTypes.inj_inGameMenuMapLoadFilters(self, superFunc, frame, ...)
    local displayCropTypes = frame.displayCropTypes
    local fruitTypeFilter = frame.fruitTypeFilter
    local filterStates = frame.filterStates
    if displayCropTypes ~= nil and fruitTypeFilter ~= nil and filterStates ~= nil then
        THUtils.call(function()
            local customData = THUtils.getDataTable(frame, self.dataKey)
            if customData == nil then
                customData = THUtils.createDataTable(frame, self.dataKey)
                customData.fruitTypeFilterStates = {}
            end
            if customData ~= nil then
                if filterStates[1] == fruitTypeFilter then
                    filterStates[1] = customData.fruitTypeFilterStates
                    for dataIndex in pairs(displayCropTypes) do
                        customData.fruitTypeFilterStates[dataIndex] = true
                    end
                    if THDebugUtil.getIsEnabled(debugFlagId) then
                        THUtils.displayMsg("Switching to custom InGameMenuMapFrame fruit type filter states [vanilla game issue fix]")
                    end
                end
            end
        end)
    end
    local function appendFunc(...)
        if displayCropTypes ~= nil and fruitTypeFilter ~= nil then
            THUtils.call(function()
                local customData = THUtils.getDataTable(frame, self.dataKey)
                if customData ~= nil then
                    local oldFruitTypeFilter = {}
                    local fruitTypeFilterStates = customData.fruitTypeFilterStates
                    for key, value in pairs(fruitTypeFilter) do
                        oldFruitTypeFilter[key] = value
                    end
                    THUtils.clearTable(fruitTypeFilter)
                    for dataIndex, cropTypeData in pairs(displayCropTypes) do
                        local fruitTypeIndex = cropTypeData.fruitTypeIndex
                        if fruitTypeIndex ~= nil and fruitTypeIndex ~= FruitType.UNKNOWN then
                            if oldFruitTypeFilter[fruitTypeIndex] == false then
                                fruitTypeFilter[fruitTypeIndex] = false
                                fruitTypeFilterStates[dataIndex] = false
                            else
                                fruitTypeFilter[fruitTypeIndex] = true
                            end
                        else
                            fruitTypeFilterStates[dataIndex] = false
                        end
                    end
                end
            end)
        end
        return ...
    end
    return appendFunc(superFunc(frame, ...))
end
function THDefaultTypes.inj_inGameMenuMapGenerateOverviewOverlay(self, superFunc, frame, ...)
    THUtils.call(function()
        updateInGameMenuMapFruitTypeFilters(self, frame)
    end)
    return superFunc(frame, ...)
end
function THDefaultTypes.inj_densityMapUtilUpdateHerbicideArea(self, superFunc, ...)
    local isFixed = false
    THUtils.call(function()
        if self.needsUpdateHerbicideAreaFix then
            local functionData = FSDensityMapUtil.functionCache.updateHerbicideArea
            if functionData == nil then
                functionData = {
                    numChangedPixels = {
                        sprayed = 0
                    },
                    totalNumPixels = {
                        sprayed = 0
                    },
                    multiModifiers = {}
                }
                local pfModData = self.thModManager:getLoadedMod("FS25_precisionFarming")
                if pfModData ~= nil then
                    functionData.defaultMultiModifiers = {}
                end
                FSDensityMapUtil.functionCache.updateHerbicideArea = functionData
                isFixed = true
            else
                if functionData.numChangedPixels ~= nil
                    and functionData.numChangedPixels.sprayed == nil
                then
                    functionData.numChangedPixels.sprayed = 0
                    isFixed = true
                end
                if functionData.totalNumPixels ~= nil
                    and functionData.totalNumPixels.sprayed == nil
                then
                    functionData.totalNumPixels.sprayed = 0
                    isFixed = true
                end
            end
        end
    end)
    local function appendFunc(...)
        if isFixed then
            THUtils.call(function()
                if THDebugUtil.getIsEnabled(debugFlagId, THDebugLevel.UPDATE) then
                    if FSDensityMapUtil.functionCache.updateHerbicideArea ~= nil then
                        THUtils.displayMsg("FSDensityMapUtil: updateHerbicideArea function parameters:")
                        THDebugUtil.printTable(FSDensityMapUtil.functionCache.updateHerbicideArea, 3)
                    end
                end
            end)
        end
        return ...
    end
    return appendFunc(superFunc(...))
end
THUtils.call(initScript)