-- Copyright 2024-2025 by Todd Hundersmarck (ThundR) 
-- All Rights Reserved

--[[

Unauthorized use and/or distribution of this work entitles
myself, the author, to unlimited free and unrestricted use,
access, and distribution of any works related to the unauthorized
user and/or distributor.

--]]

local g_thModPath = g_currentModDirectory
source(g_thModPath .. "shared/scripts/THCore.lua")
source(g_thModPath .. "shared/scripts/managers/THSpecManager.lua")
source(g_thModPath .. "shared/scripts/managers/THMapTypeManager.lua")
source(g_thModPath .. "shared/scripts/managers/THDensityMapManager.lua")
source(g_thModPath .. "shared/scripts/maps/THDensityMapController.lua")
THRowCropSystem = {}
local THRowCropSystem_mt = Class(THRowCropSystem, THCore)
THRowCropSystem.NAME = "thRowCropSystem"
THRowCropSystem.CLASS_NAME = "THRowCropSystem"
THRowCropSystem.DATA_KEY = tostring(THRowCropSystem)
THRowCropSystem.DEFAULT_BRUSH_SIZE = 0.001
THRowCropSystem.FUNCTION_CACHE_ID = {
    PREPARE_FIELD_UPDATE = "FIELDUPDATETASK.PREPARE"
}
THRowCropSystem.DEBUG_FLAG = {
    UPDATE_ROW_SPACING_AREA = "updateRowSpacingArea"
}
local debugFlagId = THCore.debugFlagId
local function initScript()
    local self = THRowCropSystem.new(THRowCropSystem.NAME)
    if self ~= nil then
        _G.g_thRowCropSystem = self
        local modPath = self.coreData.mod.path
        source(modPath .. "scripts/maps/THDensityMapUpdater.lua")
        source(modPath .. "scripts/maps/THFieldMissions.lua")
    end
end
function THRowCropSystem.new(name, customMt)
    customMt = customMt or THRowCropSystem_mt
    local self = THCore.new(name, customMt)
    if self ~= nil then
        self.fieldManager = g_fieldManager
        self.rowCropFruitTypes = {
            array = {},
            byName = {},
            byIndex = {},
            factorKeys = { "seed", "yield" }
        }
        self.infoLayers = { isValid = false, isLoadFinished = false }
        for _, otherDebugFlagId in pairs(THRowCropSystem.DEBUG_FLAG) do
            THUtils.createDebugFlagId(otherDebugFlagId)
        end
        self:initModSettings()
        g_thDensityMapController:addEventListener(self)
        return self
    end
end
function THRowCropSystem.onPreInit(self, ...)
    THUtils.callSuperClass(THRowCropSystem, "onPreInit", self, ...)
    THUtils.appendFunction("FruitTypeDesc", "updatePlantSpacing", false, false, self, THRowCropSystem.append_updatePlantSpacing)
    THUtils.overwriteFunction("FruitTypeDesc", "loadFromFoliageXMLFile", false, false, self, THRowCropSystem.overwrite_loadFromFoliageXMLFile)
    THUtils.overwriteFunction("FieldUpdateTask", "prepare", false, false, self, THRowCropSystem.overwrite_prepareFieldUpdateTask)
    local modPath = self.coreData.mod.path
    source(modPath .. "scripts/events/THToggleRowShutoffEvent.lua")
end
function THRowCropSystem.onInitTerrain(self, mission, growthSystem, terrainNode, ...)
    THUtils.callSuperClass(THRowCropSystem, "onInitTerrain", self, mission, growthSystem, terrainNode, ...)
    self:finalizeFruitTypes()
    self:loadInfoLayers(mission, terrainNode)
    if THUtils.getIsDebugEnabled() then
        THUtils.displayMsg("")
        THUtils.displayMsg("%s Information:", THRowCropSystem.CLASS_NAME)
        THUtils.printTableValues(self)
        THUtils.displayMsg("Row crop fruit types:")
        THUtils.printTable(self.rowCropFruitTypes.array, 2)
    end
end
function THRowCropSystem.onDelete(self, ...)
    g_thDensityMapController:removeEventListener(self)
    THUtils.clearTable(self.infoLayers)
    self.infoLayers.isValid = false
    self.infoLayers.isLoadFinished = false
    THUtils.callSuperClass(THRowCropSystem, "onDelete", self, ...)
end
function THRowCropSystem.loadInfoLayers(self, mission, terrainNode)
    if not self.infoLayers.isLoadFinished then
        local fruitMapSize = mission.fruitMapSize
        self.infoLayers.isValid = false
        if fruitMapSize == nil then
            THUtils.errorMsg(nil, "Fruit map not yet initialized!")
        elseif fruitMapSize < 4096 then
            THUtils.errorMsg(nil, "Fruit map size (%s) too small for row crops", fruitMapSize)
        else
            local success = true
            local rowSpacingLayer = self:createInfoLayer("rowSpacingMap", fruitMapSize, 2, "maps")
            if rowSpacingLayer == nil then
                success = false
            else
                rowSpacingLayer.dmModifier = rowSpacingLayer:createModifier()
                rowSpacingLayer.dmFilter = rowSpacingLayer:createFilter()
                rowSpacingLayer.dmNoRowFilter = rowSpacingLayer:createFilter()
                rowSpacingLayer.dmNoRowFilter:setValueCompareParams(DensityValueCompareType.EQUAL, 0)
                rowSpacingLayer.dmRowFilter1 = rowSpacingLayer:createFilter()
                rowSpacingLayer.dmRowFilter1:setValueCompareParams(DensityValueCompareType.EQUAL, 1)
                rowSpacingLayer.dmRowFilter2 = rowSpacingLayer:createFilter()
                rowSpacingLayer.dmRowFilter2:setValueCompareParams(DensityValueCompareType.EQUAL, 2)
                self.infoLayers.rowSpacing = rowSpacingLayer
            end
            if success then
                local rowShutoffLayer = self:createInfoLayer("rowShutoffMap", fruitMapSize, 1, "maps", true)
                if rowShutoffLayer == nil then
                    success = false
                else
                    rowShutoffLayer.dmModifier = rowShutoffLayer:createModifier()
                    rowShutoffLayer.dmNoShutoffFilter = rowShutoffLayer:createFilter()
                    rowShutoffLayer.dmNoShutoffFilter:setValueCompareParams(DensityValueCompareType.EQUAL, 0)
                    rowShutoffLayer.dmShutoffFilter = rowShutoffLayer:createFilter()
                    rowShutoffLayer.dmShutoffFilter:setValueCompareParams(DensityValueCompareType.EQUAL, 1)
                    self.infoLayers.rowShutoff = rowShutoffLayer
                end
            end
            if success then
                local harvestLockLayer = self:createInfoLayer("harvestLockMap", fruitMapSize, 1, "maps", true)
                if harvestLockLayer == nil then
                    success = false
                else
                    harvestLockLayer.dmModifier = harvestLockLayer:createModifier()
                    harvestLockLayer.dmNoLockFilter = harvestLockLayer:createFilter()
                    harvestLockLayer.dmNoLockFilter:setValueCompareParams(DensityValueCompareType.EQUAL, 0)
                    harvestLockLayer.dmLockFilter = harvestLockLayer:createFilter()
                    harvestLockLayer.dmLockFilter:setValueCompareParams(DensityValueCompareType.EQUAL, 1)
                    self.infoLayers.harvestLock = harvestLockLayer
                end
            end
            if success then
                local sprayLockLayer = self:createInfoLayer("sprayLockMap", fruitMapSize, 1, "maps")
                if sprayLockLayer == nil then
                    success = false
                else
                    sprayLockLayer.dmModifier = sprayLockLayer:createModifier()
                    sprayLockLayer.dmNoLockFilter = sprayLockLayer:createFilter()
                    sprayLockLayer.dmNoLockFilter:setValueCompareParams(DensityValueCompareType.EQUAL, 0)
                    sprayLockLayer.dmLockFilter = sprayLockLayer:createFilter()
                    sprayLockLayer.dmLockFilter:setValueCompareParams(DensityValueCompareType.EQUAL, 1)
                    self.infoLayers.sprayLock = sprayLockLayer
                end
            end
            if success then
                local areaMaskLayer = self:createInfoLayer("areaMaskMap", fruitMapSize, 1, "maps")
                if areaMaskLayer == nil then
                    success = false
                else
                    areaMaskLayer.dmModifier = areaMaskLayer:createModifier()
                    areaMaskLayer.dmNoAreaFilter = areaMaskLayer:createFilter()
                    areaMaskLayer.dmNoAreaFilter:setValueCompareParams(DensityValueCompareType.EQUAL, 0)
                    areaMaskLayer.dmAreaFilter = areaMaskLayer:createFilter()
                    areaMaskLayer.dmAreaFilter:setValueCompareParams(DensityValueCompareType.EQUAL, 1)
                    self.infoLayers.areaMask = areaMaskLayer
                end
            end
            if success then
                local dmGroundTypeId, groundTypeStartChannel, groundTypeNumChannels = mission.fieldGroundSystem:getDensityMapData(FieldDensityMap.GROUND_TYPE)
                local dmFieldFilter = DensityMapFilter.new(dmGroundTypeId, groundTypeStartChannel, groundTypeNumChannels)
                dmFieldFilter:setValueCompareParams(DensityValueCompareType.GREATER, 0)
                self.infoLayers.shared = { dmFieldFilter = dmFieldFilter }
                self.infoLayers.isValid = true
            else
                for _, infoLayer in pairs(self.infoLayers) do
                    if type(infoLayer) == THValueType.TABLE and type(infoLayer.delete) == THValueType.FUNCTION then
                        infoLayer:delete()
                    end
                end
            end
            self.infoLayers.isLoadFinished = true
        end
    end
    local success = self.infoLayers.isLoadFinished and self.infoLayers.isValid
    if success then
        THUtils.displayMsg("Info layers loaded successfully")
    else
        THUtils.errorMsg(nil, "Failed to load info layers")
    end
    return success
end
function THRowCropSystem.loadRowCropFruitTypeFromFoliageXML(self, fruitTypeDesc, xmlFile)
    if THUtils.argIsValid(type(fruitTypeDesc) == THValueType.TABLE, "fruitTypeDesc", fruitTypeDesc) then
        local xmlBaseKey = "foliageType"
        local fruitTypeName = xmlFile:getString(xmlBaseKey .. ".fruitType#name")
        if fruitTypeName ~= nil and fruitTypeName ~= "" then
            fruitTypeName = string.upper(fruitTypeName)
            local rcsBaseKey = xmlBaseKey .. "." .. self.coreData.xmlKey
            local success = true
            if not xmlFile:hasProperty(rcsBaseKey) then
                rcsBaseKey = xmlBaseKey .. ".thRowCropReady"
            end
            if xmlFile:hasProperty(rcsBaseKey) then
                local brushSize = xmlFile:getFloat(rcsBaseKey .. "#brushSize")
                if brushSize == nil then
                    brushSize = xmlFile:getFloat(rcsBaseKey .. "#brushWidth")
                end
                local spacingWidth = xmlFile:getFloat(rcsBaseKey .. "#spacingWidth")
                local spacingLength = xmlFile:getFloat(rcsBaseKey .. "#spacingLength")
                local numSkipRows = xmlFile:getInt(rcsBaseKey .. "#numSkipRows")
                local skipRowOffset = xmlFile:getInt(rcsBaseKey .. "#skipRowOffset")
                local skipRowsFirst = xmlFile:getBool(rcsBaseKey .. "#skipRowsFirst")
                local lockSpacing = xmlFile:getBool(rcsBaseKey .. "#lockSpacing")
                if lockSpacing == nil then
                    lockSpacing = xmlFile:getBool(rcsBaseKey .. "#lockWidth")
                end
                success = false
                if spacingWidth ~= nil and spacingWidth <= 0 then
                    THUtils.xmlErrorMsg(rcsBaseKey, nil, THMessage.VALUE_GREATER, "spacingWidth", spacingWidth, 0)
                elseif spacingLength ~= nil and spacingLength <= 0 then
                    THUtils.xmlErrorMsg(rcsBaseKey, nil, THMessage.VALUE_GREATER, "spacingLength", spacingLength, 0)
                elseif numSkipRows ~= nil and numSkipRows < 0 then
                    THUtils.xmlErrorMsg(rcsBaseKey, nil, THMessage.VALUE_GREATER_EQUAL, "numSkipRows", numSkipRows, 0)
                elseif skipRowOffset ~= nil and skipRowOffset < 0 then
                    THUtils.xmlErrorMsg(rcsBaseKey, nil, THMessage.VALUE_GREATER_EQUAL, "skipRowOffset", skipRowOffset, 0)
                elseif brushSize ~= nil and brushSize <= 0 then
                    THUtils.xmlErrorMsg(rcsBaseKey, nil, THMessage.VALUE_GREATER, "brushSize", brushSize, 0)
                else
                    local rcsFruitData = self.rowCropFruitTypes.byName[fruitTypeName]
                    if rcsFruitData == nil then
                        rcsFruitData = {
                            fruitTypeName = fruitTypeName,
                            currentValues = {},
                            defaultValues = { useRowShutoff = false, isBetweenRows = false },
                            factors = {},
                            onLoadFactors = {}
                        }
                        self.rowCropFruitTypes.byName[rcsFruitData.fruitTypeName] = rcsFruitData
                        table.insert(self.rowCropFruitTypes.array, rcsFruitData)
                    end
                    brushSize = brushSize or rcsFruitData.defaultValues.brushSize
                    spacingWidth = spacingWidth or rcsFruitData.defaultValues.spacingWidth
                    spacingLength = spacingLength or rcsFruitData.defaultValues.spacingLength
                    skipRowOffset = skipRowOffset or rcsFruitData.defaultValues.skipRowOffset
                    numSkipRows = numSkipRows or rcsFruitData.defaultValues.numSkipRows
                    skipRowsFirst = THUtils.getNoNil(skipRowsFirst, rcsFruitData.defaultValues.skipRowsFirst)
                    lockSpacing = THUtils.getNoNil(lockSpacing, rcsFruitData.lockSpacing)
                    for _, factorName in pairs(self.rowCropFruitTypes.factorKeys) do
                        local factorKey = string.format("%s.factors#%s", rcsBaseKey, factorName)
                        local factorValue = xmlFile:getFloat(factorKey)
                        if factorValue ~= nil and factorValue <= 0 then
                            THUtils.xmlErrorMsg(factorKey, nil, THMessage.VALUE_GREATER, factorName, factorValue, 0)
                        else
                            rcsFruitData.onLoadFactors[factorName] = factorValue or rcsFruitData.onLoadFactors[factorName]
                        end
                    end
                    rcsFruitData.defaultValues.brushSize = brushSize
                    rcsFruitData.defaultValues.spacingWidth = spacingWidth
                    rcsFruitData.defaultValues.spacingLength = spacingLength
                    rcsFruitData.defaultValues.skipRowOffset = skipRowOffset
                    rcsFruitData.defaultValues.numSkipRows = numSkipRows
                    rcsFruitData.defaultValues.skipRowsFirst = skipRowsFirst
                    rcsFruitData.lockSpacing = lockSpacing
                    self.rowCropFruitTypes.byName[fruitTypeName] = rcsFruitData
                    table.insert(self.rowCropFruitTypes.array, rcsFruitData)
                    success = true
                end
            end
            return success
        end
    end
    return false
end
function THRowCropSystem.finalizeFruitTypes(self)
    THUtils.clearTable(self.rowCropFruitTypes.byName)
    THUtils.clearTable(self.rowCropFruitTypes.byIndex)
    local fruitMetersPerPixel = g_thMapTypeManager:getFruitMetersPerPixel()
    if fruitMetersPerPixel <= 0 then
        THUtils.errorMsg(nil, "Cannot determine fruit map resolution required for row crops")
        THUtils.clearTable(self.rowCropFruitTypes.array)
    else
        local fruitPixelFactor = fruitMetersPerPixel / 0.125
        local rcsFruitIndex = 1
        while true do
            local rcsFruitData = self.rowCropFruitTypes.array[rcsFruitIndex]
            if rcsFruitData == nil then
                break
            end
            local thFruitTypeData = g_thMapTypeManager:getFruitType(rcsFruitData.fruitTypeName)
            if thFruitTypeData ~= nil and thFruitTypeData.index ~= FruitType.UNKNOWN and self.rowCropFruitTypes.byName[thFruitTypeData.name] == nil then
                rcsFruitData.fruitType = thFruitTypeData
                local defaultValues = rcsFruitData.defaultValues
                local currentValues = THUtils.clearTable(rcsFruitData.currentValues)
                defaultValues.fruitType = rcsFruitData
                if defaultValues.brushSize == nil then
                    defaultValues.brushSize = THRowCropSystem.DEFAULT_BRUSH_SIZE * fruitPixelFactor
                end
                defaultValues.brushSize = THUtils.clamp(defaultValues.brushSize, 0.000001, fruitMetersPerPixel)
                defaultValues.spacingWidth = math.max(0, defaultValues.spacingWidth or 0)
                defaultValues.spacingLength = math.max(0, defaultValues.spacingLength or 0)
                defaultValues.skipRowOffset = THUtils.floor(defaultValues.skipRowOffset or 0)
                defaultValues.numSkipRows = THUtils.floor(defaultValues.numSkipRows or 0)
                if defaultValues.numSkipRows <= 0 or defaultValues.skipRowOffset <= 0 then
                    defaultValues.skipRowOffset = 0
                    defaultValues.numSkipRows = 0
                    defaultValues.skipRowsFirst = false
                else
                    defaultValues.skipRowOffset = math.max(1, defaultValues.skipRowOffset)
                    defaultValues.skipRowsFirst = defaultValues.skipRowsFirst == true
                end
                local popFactorWidth = math.max(fruitMetersPerPixel, defaultValues.spacingWidth)
                local popFactorLength = math.max(fruitMetersPerPixel, defaultValues.spacingLength)
                local populationFactor = math.max(1, (popFactorWidth / fruitMetersPerPixel) * (popFactorLength / fruitMetersPerPixel))
                for _, factorName in pairs(self.rowCropFruitTypes.factorKeys) do
                    if rcsFruitData.factors[factorName] == nil or rcsFruitData.factors[factorName] <= 0 then
                        local onLoadFactor = rcsFruitData.onLoadFactors[factorName] or 1
                        rcsFruitData.factors[factorName] = populationFactor * onLoadFactor
                    end
                end
                for key, value in pairs(defaultValues) do
                    currentValues[key] = value
                end
                self.rowCropFruitTypes.byName[thFruitTypeData.name] = rcsFruitData
                self.rowCropFruitTypes.byIndex[thFruitTypeData.index] = rcsFruitData
                rcsFruitIndex = rcsFruitIndex + 1
            else
                table.remove(self.rowCropFruitTypes.array, rcsFruitIndex)
            end
        end
    end
end
function THRowCropSystem.getFieldAreaCoordinates(self, fieldId)
    local fieldDesc = self.fieldManager:getFieldById(fieldId)
    if fieldDesc ~= nil then
        local dmFieldPolygon = fieldDesc:getDensityMapPolygon()
        if dmFieldPolygon ~= nil then
            local thFieldData = THUtils.getDataTable(fieldDesc, THRowCropSystem.DATA_KEY)
            if thFieldData == nil then
                thFieldData = THUtils.createDataTable(fieldDesc, THRowCropSystem.DATA_KEY)
                thFieldData.verticesArray = dmFieldPolygon:getVerticesList()
            end
            if thFieldData ~= nil then
                local verticesArray = thFieldData.verticesArray
                local minX, minZ, maxX, maxZ = nil, nil, nil, nil
                if verticesArray ~= nil then
                    for arrayIndex = 1, #verticesArray, 2 do
                        local x = verticesArray[arrayIndex]
                        local z = verticesArray[arrayIndex + 1]
                        if x ~= nil then
                            if minX == nil or x < minX then
                                minX = x
                            end
                            if maxX == nil or x > maxX then
                                maxX = x
                            end
                        end
                        if z ~= nil then
                            if minZ == nil or z < minZ then
                                minZ = z
                            end
                            if maxZ == nil or z > maxZ then
                                maxZ = z
                            end
                        end
                    end
                end
                if minX ~= nil and maxX ~= nil and minZ ~= nil and maxZ ~= nil then
                    return minX, minZ, maxX, minZ, minX, maxZ
                end
            end
        end
    end
end
function THRowCropSystem.getRowCropFruitType(self, fruitTypeId)
    local idVarType = type(fruitTypeId)
    local rcsFruitData = nil
    if idVarType == THValueType.NUMBER then
        rcsFruitData = self.rowCropFruitTypes.byIndex[fruitTypeId]
    elseif idVarType == THValueType.STRING then
        rcsFruitData = self.rowCropFruitTypes.byName[fruitTypeId:upper()]
    elseif idVarType == THValueType.TABLE and fruitTypeId.index ~= nil then
        rcsFruitData = self.rowCropFruitTypes.byIndex[fruitTypeId.index]
    elseif fruitTypeId ~= nil then
        THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "fruitTypeId", fruitTypeId)
    end
    return rcsFruitData
end
function THRowCropSystem.getRowCropFruitTypes(self)
    return self.rowCropFruitTypes.array, #self.rowCropFruitTypes.array
end
function THRowCropSystem.getFruitTypeRowSpacing(self, fruitType, isDefaultValues)
    isDefaultValues = THUtils.validateArg(not isDefaultValues or isDefaultValues == true, "isDefaultValues", isDefaultValues, true)
    local fruitMetersPerPixel = g_thMapTypeManager:getFruitMetersPerPixel()
    if fruitMetersPerPixel > 0 then
        local rcsFruitData = self:getRowCropFruitType(fruitType)
        if rcsFruitData ~= nil then
            if isDefaultValues then
                return rcsFruitData.defaultValues
            else
                return rcsFruitData.currentValues
            end
        end
    end
end
function THRowCropSystem.setFruitTypeRowSpacing(self, fruitTypeId, spacingValues)
    local fruitMetersPerPixel = g_thMapTypeManager:getFruitMetersPerPixel()
    local rcsFruitData = self:getRowCropFruitType(fruitTypeId)
    if THUtils.argIsValid(rcsFruitData ~= nil, "fruitTypeId", fruitTypeId)
        and THUtils.argIsValid(type(spacingValues) == THValueType.TABLE, "spacingValues", spacingValues)
        and fruitMetersPerPixel > 0
    then
        local defaultValues = rcsFruitData.defaultValues
        local currentValues = rcsFruitData.currentValues
        if spacingValues == currentValues then
            return true
        end
        THUtils.clearTable(rcsFruitData.currentValues)
        for key, value in pairs(defaultValues) do
            currentValues[key] = value
        end
        local spacingWidth = spacingValues.spacingWidth
        local spacingLength = spacingValues.spacingLength
        local skipRowOffset = spacingValues.skipRowOffset
        local numSkipRows = spacingValues.numSkipRows
        if not rcsFruitData.lockSpacing then
            if spacingWidth ~= nil then
                spacingWidth = math.max(0, spacingWidth)
            else
                spacingWidth = currentValues.spacingWidth
            end
            if spacingLength ~= nil then
                spacingLength = math.max(0, spacingLength)
            else
                spacingLength = currentValues.spacingLength
            end
            if numSkipRows ~= nil then
                numSkipRows = THUtils.round(numSkipRows)
                numSkipRows = math.max(0, numSkipRows)
                if numSkipRows > 0 then
                    skipRowOffset = THUtils.round(skipRowOffset or 0)
                    skipRowOffset = math.max(1, skipRowOffset)
                else
                    skipRowOffset = 0
                end
            else
                skipRowOffset = 0
                numSkipRows = 0
            end
            if skipRowOffset ~= nil and numSkipRows ~= nil then
                currentValues.skipRowOffset = skipRowOffset
                currentValues.numSkipRows = numSkipRows
                if skipRowOffset > 0 and numSkipRows > 0 then
                    currentValues.skipRowsFirst = THUtils.getNoNil(spacingValues.skipRowsFirst, currentValues.skipRowsFirst)
                else
                    currentValues.skipRowsFirst = false
                end
            end
            currentValues.spacingWidth = spacingWidth
            currentValues.spacingLength = spacingLength
        end
        currentValues.isBetweenRows = THUtils.getNoNil(spacingValues.isBetweenRows, currentValues.isBetweenRows)
        currentValues.useRowShutoff = THUtils.getNoNil(spacingValues.useRowShutoff, currentValues.useRowShutoff)
        if THUtils.getIsDebugEnabled(debugFlagId, THDebugLevel.UPDATE) then
            THUtils.displayMsg("Spacing values set for fruit type: %s", rcsFruitData.fruitType.name)
            THUtils.displayMsg("- sent values:")
            THUtils.printTable(spacingValues)
            THUtils.displayMsg("- current values:")
            THUtils.printTable(currentValues)
            THUtils.displayMsg("- default values:")
            THUtils.printTable(defaultValues)
        end
        return true
    end
    return false
end
function THRowCropSystem.resetFruitTypeRowSpacing(self, fruitType)
    local rcsFruitData = self:getRowCropFruitType(fruitType)
    if rcsFruitData ~= nil then
        THUtils.clearTable(rcsFruitData.currentValues)
        for key, value in pairs(rcsFruitData.defaultValues) do
            rcsFruitData.currentValues[key] = value
        end
        if THUtils.getIsDebugEnabled(debugFlagId, THDebugLevel.UPDATE) then
            THUtils.displayMsg("Spacing values reset for fruit type: %s", rcsFruitData.fruitType.name)
            THUtils.printTable(rcsFruitData.currentValues)
        end
    end
end
function THRowCropSystem.getRowCropFruitTypeAroundWorldPos(self, x, z)
    local terrainNode = g_terrainNode
    local dmFunctionData = THUtils.getFunctionCache(self, "THRowCropSystem", "getRowCropFruitTypeAroundWorldPos")
    if dmFunctionData == nil then
        dmFunctionData = THUtils.createFunctionCache(self, "THRowCropSystem", "getRowCropFruitTypeAroundWorldPos", _G)
        dmFunctionData.fruitModifiers = {}
        dmFunctionData.fruitFilters = {}
    end
    local rcsFruitTypesArray, rcsNumFruitTypes = self:getRowCropFruitTypes()
    local fruitMetersPerPixel = g_thMapTypeManager:getFruitMetersPerPixel()
    if terrainNode ~= nil and rcsNumFruitTypes > 0 and fruitMetersPerPixel > 0 then
        local maxFruitType = nil
        local maxFruitArea = 0
        for arrayIndex = 1, rcsNumFruitTypes do
            local rcsFruitData = rcsFruitTypesArray[arrayIndex]
            local spacingWidth = math.max(fruitMetersPerPixel, rcsFruitData.defaultValues.spacingWidth)
            local dmFruitModifier = dmFunctionData.fruitModifiers[rcsFruitData.fruitType.index]
            local dmFruitFilter = dmFunctionData.fruitFilters[rcsFruitData.fruitType.index]
            if dmFruitModifier == nil or dmFruitFilter == nil then
                local dmFruitLayerId = rcsFruitData.fruitType.desc.terrainDataPlaneId
                local startStateChannel = rcsFruitData.fruitType.desc.startStateChannel
                local numStateChannels = rcsFruitData.fruitType.desc.numStateChannels
                if dmFruitLayerId ~= nil and startStateChannel ~= nil and numStateChannels ~= nil and numStateChannels > 0 then
                    if dmFruitModifier == nil then
                        dmFruitModifier = DensityMapModifier.new(dmFruitLayerId, startStateChannel, numStateChannels, terrainNode)
                        dmFunctionData.fruitModifiers[rcsFruitData.fruitType.index] = dmFruitModifier
                    end
                    if dmFruitFilter == nil then
                        dmFruitFilter = DensityMapFilter.new(dmFruitLayerId, startStateChannel, numStateChannels, terrainNode)
                        dmFruitFilter:setValueCompareParams(DensityValueCompareType.GREATER, 0)
                        dmFunctionData.fruitFilters[rcsFruitData.fruitType.index] = dmFruitFilter
                    end
                end
            end
            if spacingWidth > fruitMetersPerPixel
                and dmFruitModifier ~= nil and dmFruitFilter ~= nil
            then
                local minX = x - (spacingWidth * 0.5)
                local minZ = z - (spacingWidth * 0.5)
                local maxX = minX + spacingWidth
                local maxZ = minZ + spacingWidth
                dmFruitModifier:setParallelogramWorldCoords(minX, minZ, maxX, minZ, minX, maxZ, DensityCoordType.POINT_POINT_POINT)
                local _, fruitArea = dmFruitModifier:executeGet(dmFruitFilter)
                if fruitArea ~= nil and fruitArea > maxFruitArea then
                    maxFruitType = rcsFruitData
                    maxFruitArea = fruitArea
                end
            end
        end
        if maxFruitType ~= nil and maxFruitArea > 0 then
            return maxFruitType
        end
    end
end
function THRowCropSystem.getRowCropArea(self, sx, sz, wx, wz, hx, hz, numSteps, isReversed)
    local terrainNode = g_terrainNode
    local fruitMetersPerPixel = g_thMapTypeManager:getFruitMetersPerPixel()
    if THUtils.argIsValid(numSteps == nil or (type(numSteps) == THValueType.NUMBER and numSteps > 0), "numSteps", numSteps)
        and terrainNode ~= nil and fruitMetersPerPixel > 0
    then
        isReversed = THUtils.validateArg(not isReversed or isReversed == true, "isReversed", isReversed, false)
        if isReversed then
            local sx2, sz2 = sx, sz
            local x4 = hx + (wx - sx)
            local z4 = hz + (wz - sz)
            sx = hx
            sz = hz
            wx = x4
            wz = z4
            hx = sx2
            hz = sz2
        end
        local sy = getTerrainHeightAtWorldPos(terrainNode, sx, 0, sz)
        local hy = getTerrainHeightAtWorldPos(terrainNode, hx, 0, hz)
        local workLength = THUtils.getVector3Distance(sx, sy, sz, hx, hy, hz)
        local dwx = wx - sx
        local dwz = wz - sz
        local dhx = hx - sx
        local dhz = hz - sz
        if workLength >= fruitMetersPerPixel then
            local alignedLength = workLength
            local isAreaChanged = false
            if numSteps == nil then
                alignedLength = math.max(fruitMetersPerPixel, alignedLength)
            else
                alignedLength = math.max(fruitMetersPerPixel, fruitMetersPerPixel * numSteps)
            end
            if math.abs(alignedLength - workLength) >= 0.000001 then
                dhx = dhx * (alignedLength / workLength)
                dhz = dhz * (alignedLength / workLength)
                isAreaChanged = true
            end
            if isAreaChanged then
                wx = sx + dwx
                wz = sz + dwz
                hx = sx + dhx
                hz = sz + dhz
            end
        end
    end
    return sx, sz, wx, wz, hx, hz
end
function THRowCropSystem.updateRowSpacingArea(self, sx, sz, wx, wz, hx, hz, spacingValues, rowCoords, noRowShutoff, limitToField)
    if THUtils.argIsValid(type(spacingValues) == THValueType.TABLE, "spacingValues", spacingValues)
        and THUtils.argIsValid(rowCoords == nil or type(rowCoords) == THValueType.TABLE, "rowCoords", rowCoords)
        and THUtils.argIsValid(type(sx) == THValueType.NUMBER, "sx", sx)
        and THUtils.argIsValid(type(sz) == THValueType.NUMBER, "sz", sz)
        and THUtils.argIsValid(type(wx) == THValueType.NUMBER, "wx", wx)
        and THUtils.argIsValid(type(wz) == THValueType.NUMBER, "wz", wz)
        and THUtils.argIsValid(type(hx) == THValueType.NUMBER, "hx", hx)
        and THUtils.argIsValid(type(hz) == THValueType.NUMBER, "hz", hz)
    then
        noRowShutoff = THUtils.validateArg(not noRowShutoff or noRowShutoff == true, "noRowShutoff", noRowShutoff, true)
        limitToField = THUtils.validateArg(not limitToField or limitToField == true, "limitToField", limitToField, false)
        local terrainNode = g_terrainNode
        local fruitMetersPerPixel = g_thMapTypeManager:getFruitMetersPerPixel()
        local infoLayerData = self.infoLayers
        if infoLayerData.isValid
            and terrainNode ~= nil and fruitMetersPerPixel > 0
        then
            local sy = getTerrainHeightAtWorldPos(terrainNode, sx, 0, sz)
            local wy = getTerrainHeightAtWorldPos(terrainNode, wx, 0, wz)
            local hy = getTerrainHeightAtWorldPos(terrainNode, hx, 0, hz)
            local osx, osz = sx, sz
            local owx, owz = wx, wz
            local ohx, ohz = hx, hz
            local workWidth = THUtils.getVector3Distance(sx, sy, sz, wx, wy, wz)
            local workLength = THUtils.getVector3Distance(sx, sy, sz, hx, hy, hz)
            local brushSize = spacingValues.brushSize or THRowCropSystem.DEFAULT_BRUSH_SIZE
            local spacingWidth = math.max(fruitMetersPerPixel, spacingValues.spacingWidth or 0)
            local spacingLength = math.max(fruitMetersPerPixel, spacingValues.spacingLength or 0)
            local skipRowOffset = spacingValues.skipRowOffset or 0
            local numSkipRows = spacingValues.numSkipRows or 0
            local skipRowsFirst = THUtils.getNoNil(spacingValues.skipRowsFirst, false)
            local isBetweenRows = THUtils.getNoNil(spacingValues.isBetweenRows, false)
            local useRowShutoff = THUtils.getNoNil(spacingValues.useRowShutoff, false)
            local thSpacingLayer = infoLayerData.rowSpacing
            local dmSpacingModifier = thSpacingLayer.dmModifier
            local dmRowMask = 1
            local dmFruitMask = 2
            local thRowShutoffLayer = self.infoLayers.rowShutoff
            local dmNoShutoffFilter = nil
            if useRowShutoff and not noRowShutoff then
                dmNoShutoffFilter = thRowShutoffLayer.dmNoShutoffFilter
            end
            local dmFieldFilter = nil
            if limitToField then
                dmFieldFilter = infoLayerData.shared.dmFieldFilter
            end
            dmSpacingModifier:setParallelogramWorldCoords(sx, sz, wx, wz, hx, hz, DensityCoordType.POINT_POINT_POINT)
            dmSpacingModifier:executeSet(0)
            local numStepsWidth = THUtils.round(workWidth / spacingWidth)
            local numStepsLength = 0
            if rowCoords == nil then
                numStepsLength = THUtils.round(workLength / spacingLength)
            else
                numStepsLength = THUtils.round(workLength / fruitMetersPerPixel)
            end
            numStepsLength = math.max(1, numStepsLength)
            if numStepsWidth <= 0 then
                if skipRowOffset == 0 or not skipRowsFirst then
                    if dmNoShutoffFilter ~= nil then
                        dmSpacingModifier:executeSet(dmFruitMask, dmNoShutoffFilter, dmFieldFilter)
                    else
                        dmSpacingModifier:executeSet(dmFruitMask, dmFieldFilter)
                    end
                end
            elseif workLength <= 0 then
                THUtils.printDebugError(nil, false, "Work area is too small")
            else
                local dwx = wx - sx
                local dwz = wz - sz
                local dhx = hx - sx
                local dhz = hz - sz
                local stepWX = dwx / numStepsWidth
                local stepWZ = dwz / numStepsWidth
                local stepHX = dhx / numStepsLength
                local stepHZ = dhz / numStepsLength
                local bwx = dwx * (brushSize / workWidth)
                local bwz = dwz * (brushSize / workWidth)
                local bhx = stepHX
                local bhz = stepHZ
                local startX = osx
                local startZ = osz
                if isBetweenRows then
                    numStepsWidth = numStepsWidth + 1
                else
                    startX = osx + (stepWX * 0.5)
                    startZ = osz + (stepWZ * 0.5)
                end
                local numPaintedRows = 0
                local numSkippedRows = 0
                if skipRowOffset > 0 and skipRowsFirst then
                    numPaintedRows = skipRowOffset
                end
                local isPreciseMode = rowCoords ~= nil and numStepsLength > 1 and spacingLength > fruitMetersPerPixel
                local isDebugEnabled = THUtils.getIsDebugEnabled(THRowCropSystem.DEBUG_FLAG.UPDATE_ROW_SPACING_AREA)
                local isDebugUpdateEnabled = isDebugEnabled and THUtils.getDebugLevel() >= THDebugLevel.UPDATE
                if isDebugUpdateEnabled then
                    THUtils.displayMsg("THRowCropSystem.updateRowSpacingArea:")
                    THUtils.displayMsg("- workWidth: %s", workWidth)
                    THUtils.displayMsg("- workLength: %s", workLength)
                    THUtils.displayMsg("- numStepsWidth: %s", numStepsWidth)
                    THUtils.displayMsg("- numStepsLength: %s", numStepsLength)
                    THUtils.displayMsg("- isPreciseMode: %s", isPreciseMode)
                    THUtils.displayMsg("")
                    THUtils.displayMsg("- sx, sz: %0.3f, %0.3f", osx, osz)
                    THUtils.displayMsg("- wx, wz: %0.3f, %0.3f", owx, owz)
                    THUtils.displayMsg("- hx, hz: %0.3f, %0.3f", ohx, ohz)
                    THUtils.displayMsg("")
                    THUtils.printTable(spacingValues, 1)
                end
                local function addDrawDebugLines(psx, psz, pwx, pwz, phx, phz, pr, pg, pb)
                    local vsy = getTerrainHeightAtWorldPos(terrainNode, psx, 0, psz)
                    local vwy = getTerrainHeightAtWorldPos(terrainNode, pwx, 0, pwz)
                    local vhy = getTerrainHeightAtWorldPos(terrainNode, phx, 0, phz)
                    g_thEventManager:addDrawTask(10000, function()
                        DebugUtil.drawDebugAreaRectangle(psx, vsy + 0.5, psz, pwx, vwy + 0.5, pwz, phx, vhy + 0.5, phz, false, pr, pg, pb)
                    end)
                end
                local function paintRow(psx, psy, psz, pwx, pwz, phx, phz)
                    local vMaskValue = (isPreciseMode and dmFruitMask) or dmRowMask
                    dmSpacingModifier:setParallelogramWorldCoords(psx, psz, pwx, pwz, phx, phz, DensityCoordType.POINT_POINT_POINT)
                    if isDebugEnabled then
                        local _, dmChangedArea = 0, 0
                        if dmNoShutoffFilter ~= nil then
                            _, dmChangedArea = dmSpacingModifier:executeSetWithStats(vMaskValue, dmNoShutoffFilter, dmFieldFilter)
                        else
                            _, dmChangedArea = dmSpacingModifier:executeSetWithStats(vMaskValue, dmFieldFilter)
                        end
                        if dmChangedArea ~= nil and dmChangedArea > 0 then
                            if vMaskValue == dmFruitMask then
                                local vsy = getTerrainHeightAtWorldPos(terrainNode, psx, 0, psz)
                                local vwy = getTerrainHeightAtWorldPos(terrainNode, pwx, 0, pwz)
                                local vhy = getTerrainHeightAtWorldPos(terrainNode, phx, 0, phz)
                                DebugUtil.drawDebugAreaRectangle(psx, vsy + 0.5, psz, pwx, vwy + 0.5, pwz, phx, vhy + 0.5, phz, false, 1, 1, 1)
                                drawDebugLine(psx, psy - 1, psz, 1, 1, 1, psx, psy + 2, psz, 1, 1, 1)
                            else
                                addDrawDebugLines(psx, psz, pwx, pwz, phx, phz, 1, 1, 1)
                            end
                        end
                    else
                        if dmNoShutoffFilter ~= nil then
                            dmSpacingModifier:executeSet(vMaskValue, dmNoShutoffFilter, dmFieldFilter)
                        else
                            dmSpacingModifier:executeSet(vMaskValue, dmFieldFilter)
                        end
                    end
                end
                for stepWidth = 1, numStepsWidth do
                    local isRowSkipped = skipRowOffset > 0 and numPaintedRows >= skipRowOffset
                    if isRowSkipped then
                        numSkippedRows = numSkippedRows + 1
                        if numSkippedRows >= numSkipRows then
                            numPaintedRows = 0
                            numSkippedRows = 0
                        end
                    else
                        if isPreciseMode then
                            for stepLength = 1, numStepsLength do
                                sx = startX + (stepWX * (stepWidth - 1)) + (stepHX * (stepLength - 1))
                                sz = startZ + (stepWZ * (stepWidth - 1)) + (stepHZ * (stepLength - 1))
                                sy = getTerrainHeightAtWorldPos(terrainNode, sx, 0, sz)
                                local lastDistance = nil
                                local rowStepCoords = rowCoords[stepWidth]
                                if rowStepCoords ~= nil then
                                    local lastSX, lastSY, lastSZ = THUtils.unpack(rowStepCoords)
                                    lastDistance = THUtils.getVector3Distance(lastSX, lastSY, lastSZ, sx, sy, sz)
                                end
                                if lastDistance == nil or lastDistance >= spacingLength then
                                    wx = sx + bwx
                                    wz = sz + bwz
                                    hx = sx + bhx
                                    hz = sz + bhz
                                    paintRow(sx, sy, sz, wx, wz, hx, hz)
                                    if rowStepCoords == nil then
                                        rowStepCoords = THUtils.pack(sx, sy, sz)
                                        rowCoords[stepWidth] = rowStepCoords
                                    else
                                        rowStepCoords[1] = sx
                                        rowStepCoords[2] = sy
                                        rowStepCoords[3] = sz
                                    end
                                end
                            end
                        else
                            sx = startX + (stepWX * (stepWidth - 1))
                            sz = startZ + (stepWZ * (stepWidth - 1))
                            local alignedSX = g_thMapTypeManager:getFruitPixelAlignedValue(sx)
                            local alignedSZ = g_thMapTypeManager:getFruitPixelAlignedValue(sz)
                            if math.abs(alignedSX - sx) <= brushSize
                                or math.abs(alignedSZ - sz) <= brushSize
                            then
                                sx = sx - bwx - 0.001
                                sz = sz - bwz - 0.001
                            end
                            sy = getTerrainHeightAtWorldPos(terrainNode, sx, 0, sz)
                            wx = sx + bwx
                            wz = sz + bwz
                            hx = sx + dhx
                            hz = sz + dhz
                            paintRow(sx, sy, sz, wx, wz, hx, hz)
                        end
                        numPaintedRows = numPaintedRows + 1
                    end
                end
                if not isPreciseMode then
                    if numStepsLength > 1 and spacingLength > fruitMetersPerPixel then
                        startX = osx
                        startZ = osz
                        bhx = dhx * (brushSize / workLength)
                        bhz = dhz * (brushSize / workLength)
                        for stepLength = 1, numStepsLength do
                            sx = startX + (stepHX * (stepLength - 1))
                            sz = startZ + (stepHZ * (stepLength - 1))
                            wx = sx + dwx
                            wz = sz + dwz
                            hx = sx + bhx
                            hz = sz + bhz
                            dmSpacingModifier:setParallelogramWorldCoords(sx, sz, wx, wz, hx, hz, DensityCoordType.POINT_POINT_POINT)
                            if isDebugEnabled then
                                local _, dmChangedArea = dmSpacingModifier:executeSetWithStats(dmFruitMask, thSpacingLayer.dmRowFilter1)
                                if dmChangedArea ~= nil and dmChangedArea > 0 then
                                    addDrawDebugLines(sx, sz, wx, wz, hx, hz, 1, 0, 0)
                                end
                            else
                                dmSpacingModifier:executeSet(dmFruitMask, thSpacingLayer.dmRowFilter1)
                            end
                        end
                    else
                        dmSpacingModifier:setParallelogramWorldCoords(osx, osz, owx, owz, ohx, ohz, DensityCoordType.POINT_POINT_POINT)
                        dmSpacingModifier:executeSet(dmFruitMask, thSpacingLayer.dmRowFilter1)
                    end
                end
            end
        end
    end
end
function THRowCropSystem.updateFruitRowSpacingArea(self, fruitType, sx, sz, wx, wz, hx, hz, rowCoords, isDefaultValues, noRowShutoff, limitToField)
    local spacingValues = self:getFruitTypeRowSpacing(fruitType, isDefaultValues)
    if spacingValues ~= nil then
        self:updateRowSpacingArea(sx, sz, wx, wz, hx, hz, spacingValues, rowCoords, noRowShutoff, limitToField)
    end
end
function THRowCropSystem.dmOnMultiModifierAddExecute(self, parent, dmFunctionData, superFunc, callingParams, returnParams)
    if dmFunctionData.id == THRowCropSystem.FUNCTION_CACHE_ID.PREPARE_FIELD_UPDATE then
        self:dmOnMultiModifierAddExecuteFieldUpdateTask(parent, dmFunctionData, superFunc, callingParams, returnParams)
    end
end
function THRowCropSystem.dmOnMultiModifierAddExecuteFieldUpdateTask(self, parent, dmFunctionData, superFunc, callingParams, returnParams)
    if self.infoLayers.isValid then
        local thSpacingLayer = self.infoLayers.rowSpacing
        local fieldUpdateTask = dmFunctionData.parent
        local rcsFruitData = dmFunctionData.fruitType
        local dmMultiModifier = fieldUpdateTask.multiModifier
        local dmFuncType = callingParams.dmFuncType
        local dmModifier = callingParams.dmModifier
        local dmCallingValue = callingParams.value
        local dmFruitModifier = dmFunctionData.dmFruitModifier
        local dmFilter1 = fieldUpdateTask.filter1
        local dmFilter2 = fieldUpdateTask.filter2
        local dmFilter3 = fieldUpdateTask.filter3
        if rcsFruitData ~= nil and parent == dmMultiModifier and dmModifier ~= nil then
            if dmFuncType == THDMFuncType.SET then
                if dmCallingValue > 0 then
                    local dmModifierLayerId = g_thDensityMapController:getDensityMapLayerId(dmModifier)
                    local thModifierFruitData = g_thMapTypeManager:getFruitTypeByDataPlaneId(dmModifierLayerId)
                    if thModifierFruitData ~= nil and thModifierFruitData.index == rcsFruitData.fruitType.index then
                        parent:addExecuteSet(0, dmFruitModifier, dmFilter1, dmFilter2, dmFilter3)
                        self:dmExecuteAreaMask(dmFunctionData, callingParams, thSpacingLayer.dmRowFilter2, true)
                    end
                end
            end
        end
    end
end
function THRowCropSystem.dmExecuteAreaMask(self, dmFunctionData, callingParams, dmAddedFilter, isMultiModifier)
    if THUtils.argIsValid(type(dmFunctionData) == THValueType.TABLE, "dmFunctionData", dmFunctionData) and THUtils.argIsValid(type(callingParams) == THValueType.TABLE, "callingParams", callingParams) and
        THUtils.argIsValid(type(dmAddedFilter) == THValueType.USERDATA, "dmAddedFilter", dmAddedFilter) and
        THUtils.argIsValid(not isMultiModifier or isMultiModifier == true, "isMultiModifier", isMultiModifier) and self.infoLayers.isValid then
        local dmCallingFilters = callingParams.dmFilters
        if dmCallingFilters.n <= 1 then
            THUtils.addPackedEntry(dmCallingFilters, dmAddedFilter)
        else
            local thAreaMaskLayer = self.infoLayers.areaMask
            local dmAreaModifier = thAreaMaskLayer.dmModifier
            local dmAreaFilter = thAreaMaskLayer.dmAreaFilter
            local dmParent = callingParams.dmParent
            local dmModifier = dmParent
            if isMultiModifier then
                dmModifier = callingParams.dmModifier
            end
            local dmModifierLayerId = g_thDensityMapController:getDensityMapLayerId(dmModifier)
            local filterIndex = 1
            local foundFilterIndex = nil
            while true do
                local dmCallingFilter = dmCallingFilters[filterIndex]
                if type(dmCallingFilter) ~= THValueType.USERDATA then
                    break
                end
                local dmFilterLayerId = g_thDensityMapController:getDensityMapLayerId(dmCallingFilter)
                if dmFilterLayerId == nil or dmFilterLayerId ~= dmModifierLayerId then
                    foundFilterIndex = filterIndex
                    break
                end
                filterIndex = filterIndex + 1
            end
            foundFilterIndex = foundFilterIndex or (filterIndex - 1)
            if foundFilterIndex <= 0 then
                THUtils.addPackedEntry(dmCallingFilters, dmAddedFilter, 1)
            else
                local dmFoundFilter = dmCallingFilters[foundFilterIndex]
                if isMultiModifier then
                    dmParent:addExecuteSet(0, dmAreaModifier)
                    dmParent:addExecuteSet(1, dmAreaModifier, dmFoundFilter, dmAddedFilter)
                else
                    dmAreaModifier:executeSet(0)
                    dmAreaModifier:executeSet(1, dmFoundFilter, dmAddedFilter)
                end
                dmCallingFilters[foundFilterIndex] = dmAreaFilter
                if foundFilterIndex > 3 then
                    if THUtils.getIsDebugEnabled() then
                        local dmLayerName = callingParams.dmFuncName
                        if dmModifierLayerId ~= nil then
                            dmLayerName = getName(dmModifierLayerId)
                        end
                        THUtils.errorMsg(true, "Too many filters for density modifier: %s", dmLayerName)
                    end
                end
            end
        end
        return true
    end
    return false
end
function THRowCropSystem.dmExecuteHarvestLock(self, dmFunctionData, callingParams, dmSearchFilter)
    if THUtils.argIsValid(type(dmFunctionData) == THValueType.TABLE, "dmFunctionData", dmFunctionData) and THUtils.argIsValid(type(callingParams) == THValueType.TABLE, "callingParams", callingParams) and
        THUtils.argIsValid(dmSearchFilter == nil or type(dmSearchFilter) == THValueType.USERDATA, "dmSearchFilter", dmSearchFilter) and self.infoLayers.isValid then
        local thHarvestLockLayer = self.infoLayers.harvestLock
        local dmHarvestNoLockFilter = thHarvestLockLayer.dmNoLockFilter
        local dmCallingFilters = callingParams.dmFilters
        local searchFilterIndex = nil
        if dmSearchFilter == nil or dmCallingFilters.n <= 0 then
            searchFilterIndex = 0
        else
            for filterIndex = 1, 3 do
                local dmCallingFilter = dmCallingFilters[filterIndex]
                if dmCallingFilter == dmSearchFilter then
                    searchFilterIndex = filterIndex
                    break
                end
            end
        end
        if searchFilterIndex ~= nil then
            THUtils.safePack(dmCallingFilters, dmHarvestNoLockFilter)
            return true
        end
    end
    return false
end
function THRowCropSystem.append_updatePlantSpacing(self, superFunc, fruitTypeDesc, ...)
    if fruitTypeDesc ~= nil and fruitTypeDesc.plantSpacing ~= nil then
        local rcsFruitData = self:getRowCropFruitType(fruitTypeDesc.index)
        if rcsFruitData ~= nil and rcsFruitData.defaultValues.spacingWidth > 0 then
            fruitTypeDesc.plantSpacing = rcsFruitData.defaultValues.spacingWidth
        end
    end
end
function THRowCropSystem.overwrite_loadFromFoliageXMLFile(self, superFunc, fruitTypeDesc, xmlFilename, ...)
    local XMLFile = g_thGlobalEnv.XMLFile
    local oldLoadXMLFile = XMLFile.load
    local prependSuccess = false
    if type(fruitTypeDesc) == THValueType.TABLE and type(xmlFilename) == THValueType.STRING and xmlFilename ~= "" then
        THUtils.pcall(function()
            XMLFile.load = function(pXMLName, pXMLFilename, ...)
                local function vAppendFunc(rXMLFile, ...)
                    if pXMLFilename == xmlFilename and rXMLFile ~= nil then
                        THUtils.pcall(function()
                            local wrappedXML = THUtils.wrapXMLFile(rXMLFile)
                            if wrappedXML ~= nil then
                                self:loadRowCropFruitTypeFromFoliageXML(fruitTypeDesc, wrappedXML)
                            end
                        end)
                    end
                    return rXMLFile, ...
                end
                return vAppendFunc(oldLoadXMLFile(pXMLName, pXMLFilename, ...))
            end
            prependSuccess = true
        end)
    end
    local function appendFunc(...)
        if prependSuccess then
            XMLFile.load = oldLoadXMLFile
        end
        return ...
    end
    return appendFunc(superFunc(fruitTypeDesc, xmlFilename, ...))
end
function THRowCropSystem.overwrite_prepareFieldUpdateTask(self, superFunc, parent, ...)
    local terrainNode = g_terrainNode or 0
    local prependSuccess = false
    THUtils.pcall(function()
        if self.infoLayers.isValid then
            local thRowShutoffLayer = self.infoLayers.rowShutoff
            local thHarvestLockLayer = self.infoLayers.harvestLock
            local fieldId = parent.fieldId
            local dmFieldPolygon = parent.area
            if fieldId ~= nil and dmFieldPolygon ~= nil then
                thRowShutoffLayer:resetPolygon(dmFieldPolygon)
                thHarvestLockLayer:resetPolygon(dmFieldPolygon)
                local rcsFruitData = self:getRowCropFruitType(parent.fruitTypeIndex)
                local dmMultiModifier = parent.multiModifier
                local sx, sz, wx, wz, hx, hz = self:getFieldAreaCoordinates(fieldId)
                if rcsFruitData ~= nil and dmMultiModifier ~= nil and sx ~= nil then
                    local terrainDataPlaneId = rcsFruitData.fruitType.desc.terrainDataPlaneId
                    local startStateChannel = rcsFruitData.fruitType.desc.startStateChannel
                    local numStateChannels = rcsFruitData.fruitType.desc.numStateChannels
                    if terrainNode ~= nil
                        and terrainDataPlaneId ~= nil and startStateChannel ~= nil
                        and numStateChannels ~= nil and numStateChannels > 0
                    then
                        local dmFunctionData = THUtils.getFunctionCache(self, "FieldUpdateTask", "prepare")
                        if dmFunctionData == nil then
                            dmFunctionData = THUtils.createFunctionCache(self, "FieldUpdateTask", "prepare")
                        end
                        local dmFruitModifier = dmFunctionData.dmFruitModifier
                        if dmFruitModifier == nil then
                            dmFruitModifier = DensityMapModifier.new(terrainDataPlaneId, startStateChannel, numStateChannels, terrainNode)
                            dmFunctionData.dmFruitModifier = dmFruitModifier
                        else
                            dmFruitModifier:resetDensityMapAndChannels(terrainDataPlaneId, startStateChannel, numStateChannels)
                        end
                        dmFruitModifier:setNewTypeIndexMode(DensityIndexCompareMode.ZERO)
                        dmFunctionData.parent = parent
                        dmFunctionData.fruitType = rcsFruitData
                        dmFunctionData.dmFruitModifier = dmFruitModifier
                        self:updateFruitRowSpacingArea(rcsFruitData.fruitType.index, sx, sz, wx, wz, hx, hz, nil, true, true, true)
                        parent:setGroundAngle(0)
                        g_thDensityMapController:setActiveFunctionData(dmFunctionData)
                        prependSuccess = true
                    end
                end
            end
        end
    end)
    local function appendFunc(...)
        if prependSuccess then
            THUtils.pcall(function()
                g_thDensityMapController:restoreLastActiveFunctionData()
            end)
        end
        return ...
    end
    return appendFunc(superFunc(parent, ...))
end
THUtils.pcall(initScript)