-- Copyright 2024-2025 by Todd Hundersmarck (ThundR) 
-- All Rights Reserved

--[[

Unauthorized use and/or distribution of this work entitles
myself, the author, to unlimited free and unrestricted use,
access, and distribution of any works related to the unauthorized
user and/or distributor.

--]]

THPFModule = {}
local THPFModule_mt = Class(THPFModule)
THPFModule.NAME = "thPFModule"
THPFModule.CLASS_NAME = "THPFModule"
THPFModule.DATA_KEY = tostring(THPFModule)
THPFModule.FUNCTION_CACHE_ID = {
    UPDATE_COVER_AREA = "COVERMAP.PREUPDATECOVERAREA",
    NITROGEN_HARVEST_COVER_UPDATE = "NITROGENMAP.ONHARVESTCOVERUPDATE"
}
THPFModule.DEBUG_FLAG = {
    MODULE = THPFModule.CLASS_NAME,
    COVER_MAP = "CoverMap"
}
local function initScript()
    local self = THPFModule.new()
    if self ~= nil then
        _G.g_thPFModule = self
    end
end
function THPFModule.new(customMt)
    customMt = customMt or THPFModule_mt
    local pfModData = g_thModManager:getActiveMod("FS25_precisionFarming")
    if pfModData ~= nil then
        local self = setmetatable({}, customMt)
        self.pfMod = pfModData
        self.precisionFarming = self.pfMod.environment.g_precisionFarming
        self.infoLayers = {
            isValid = false,
            isLoadFinished = false
        }
        for _, debugFlagId in pairs(THPFModule.DEBUG_FLAG) do
            THUtils.createDebugFlagId(debugFlagId)
        end
        g_thEventManager:addEventListener(self)
        g_thDensityMapController:addEventListener(self)
        return self
    end
end
function THPFModule.onPreInit(self, ...)
    THUtils.callSuperClass(THPFModule, "onPreInit", self, ...)
    THUtils.overwriteFunction(THRowCropSystem, "loadInfoLayers", false, false, self, THPFModule.overwrite_loadInfoLayers)
    THUtils.overwriteFunction(FSDensityMapUtil, "updateDestroyCommonArea", false, false, self, THPFModule.overwrite_updateDestroyCommonArea)
    THUtils.overwriteFunction(FieldUpdateTask, "prepare", false, false, self, THPFModule.overwrite_prepareFieldUpdateTask)
    local CoverMap = self.pfMod.environment.CoverMap
end
function THPFModule.onDelete(self, ...)
    g_thEventManager:removeEventListener(self)
    g_thDensityMapController:removeEventListener(self)
    THUtils.clearTable(self.infoLayers)
    self.infoLayers.isValid = false
    self.infoLayers.isLoadFinished = false
    THUtils.callSuperClass(THPFModule, "onDelete", self, ...)
end
function THPFModule.dmOnModifierExecute(self, parent, dmFunctionData, superFunc, callingParams, returnParams)
    if dmFunctionData.id == THPFModule.FUNCTION_CACHE_ID.UPDATE_COVER_AREA then
        self:dmOnModifierExecuteCoverArea(parent, dmFunctionData, superFunc, callingParams, returnParams)
    end
end
function THPFModule.dmOnModifierExecuteCoverArea(self, parent, dmFunctionData, superFunc, callingParams, returnParams)
    if self.infoLayers.isValid then
        local pfCoverMap = dmFunctionData.target
        local dmUpdateModifiers = pfCoverMap.densityMapModifiersUpdate
        local fruitTypesArray = dmFunctionData.fruitTypesArray
        if dmUpdateModifiers ~= nil and fruitTypesArray ~= nil then
            local dmFuncType = callingParams.dmFuncType
            local dmCallingFilters = callingParams.dmFilters
            local dmCallingValue = callingParams.value
            local dmBaseLockModifier = dmUpdateModifiers.lockModifier
            local dmBaseFruitFilterHarvestState = dmUpdateModifiers.fruitFilterHarvestState
            local isDebugAllEnabled = THUtils.getIsDebugEnabled(THPFModule.DEBUG_FLAG.COVER_MAP, THDebugLevel.ALL)
            if isDebugAllEnabled then
                THUtils.displayMsg("THPFModule.dmOnModifierExecuteCoverArea:")
                THUtils.displayMsg("")
                THUtils.displayMsg("- coverMap:")
                THUtils.printTable(pfCoverMap, 1)
                THUtils.displayMsg("- dmFunctionData:")
                THUtils.printTable(dmFunctionData, 1)
                THUtils.displayMsg("- callingParams:")
                THUtils.printTable(callingParams, 1)
            end
            if dmFuncType == THDMFuncType.SET then
                if parent == dmBaseLockModifier then
                    if dmCallingValue == 1 then
                        if dmCallingFilters[1] == dmBaseFruitFilterHarvestState then
                            local dmLayerId = g_thDensityMapController:getDensityMapLayerId(dmBaseFruitFilterHarvestState)
                            local thFruitTypeData = g_thMapTypeManager:getFruitTypeByDataPlaneId(dmLayerId)
                            if thFruitTypeData ~= nil and g_thRowCropSystem:getRowCropFruitType(thFruitTypeData.index) ~= nil then
                                local _, dmChangedArea = self:dmModifierExecuteHarvestCoverUpdate(dmFunctionData, callingParams, returnParams, dmBaseFruitFilterHarvestState)
                                if dmChangedArea ~= nil and dmChangedArea > 0 then
                                    if dmChangedArea > dmFunctionData.maxChangedArea
                                        or dmFunctionData.fruitTypeIndex == nil or dmFunctionData.fruitTypeIndex == FruitType.UNKNOWN
                                    then
                                        dmFunctionData.fruitTypeIndex = thFruitTypeData.index
                                        if dmChangedArea > dmFunctionData.maxChangedArea then
                                            dmFunctionData.maxChangedArea = dmChangedArea
                                            dmFunctionData.lastHarvestLockArea = dmChangedArea
                                        end
                                    end
                                end
                            end
                        end
                    end
                end
            end
        end
    end
end
function THPFModule.dmModifierExecuteHarvestCoverUpdate(self, dmFunctionData, callingParams, returnParams, ...)
    if self.infoLayers.isValid then
        local thAreaMaskLayer = g_thRowCropSystem.infoLayers.areaMask
        local dmAreaModifier = thAreaMaskLayer.dmModifier
        local thPFAreaMaskLayer = self.infoLayers.areaMask
        local dmPFAreaModifier = thPFAreaMaskLayer.dmModifier
        local dmPFAreaFilter = thPFAreaMaskLayer.dmFilter
        local thPFHarvestLockLayer = self.infoLayers.harvestLock
        local dmPFHarvestLockFilter = thPFHarvestLockLayer.dmFilter
        local dmCallingFilters = callingParams.dmFilters
        dmAreaModifier:executeSet(0)
        local dmSumPixels, dmChangedArea, dmTotalArea = 0, 0, 0
        local numCustomFilters = select("#", ...) or 0
        if numCustomFilters <= 0 then
            dmSumPixels, dmChangedArea, dmTotalArea = dmAreaModifier:executeSetWithStats(1, THUtils.unpack(dmCallingFilters))
        else
            dmSumPixels, dmChangedArea, dmTotalArea = dmAreaModifier:executeSetWithStats(1, ...)
        end
        if dmChangedArea ~= nil then
            dmPFAreaModifier:executeSet(0)
            if dmChangedArea > 0 then
                dmSumPixels, dmChangedArea, dmTotalArea = dmPFAreaModifier:executeSetWithStats(1)
                if dmChangedArea ~= nil and dmChangedArea > 0 then
                    local dmSumPixels2, dmChangedArea2, dmTotalArea2 = dmPFAreaModifier:executeSetWithStats(0, dmPFHarvestLockFilter)
                    if dmChangedArea2 ~= nil then
                        dmTotalArea = dmTotalArea2
                        dmChangedArea = math.min(dmTotalArea2, dmTotalArea2 - dmChangedArea2)
                        dmSumPixels = math.min(dmTotalArea2, dmTotalArea2 - dmSumPixels2)
                    end
                end
            end
            THUtils.safePack(dmCallingFilters, dmPFAreaFilter)
            return dmSumPixels, dmChangedArea, dmTotalArea
        end
    end
end
function THPFModule.overwrite_loadInfoLayers(self, superFunc, parent, mission, terrainNode, ...)
    local function appendFunc(rSuccess, ...)
        THUtils.pcall(function()
            if not self.infoLayers.isLoadFinished then
                local fruitMapSize = mission.fruitMapSize
                local precisionFarming = self.precisionFarming
                self.infoLayers.isValid = false
                if rSuccess and fruitMapSize > 4096 and parent.infoLayers.isValid
                    and precisionFarming ~= nil and precisionFarming.soilMap ~= nil
                then
                    local pfSoilMap = precisionFarming.soilMap
                    local pfSoilMapSize = math.max(pfSoilMap.sizeX, pfSoilMap.sizeY)
                    local success = true
                    if success then
                        local pfAreaMaskLayer = g_thRowCropSystem:createInfoLayer("pfAreaMaskMap", pfSoilMapSize, 1, "modules/precisionFarming/maps")
                        if pfAreaMaskLayer == nil then
                            success = false
                        else
                            pfAreaMaskLayer.dmModifier = pfAreaMaskLayer:createModifier()
                            pfAreaMaskLayer.dmFilter = pfAreaMaskLayer:createFilter()
                            pfAreaMaskLayer.dmFilter:setValueCompareParams(DensityValueCompareType.EQUAL, 1)
                            self.infoLayers.areaMask = pfAreaMaskLayer
                        end
                    end
                    if success then
                        local pfHarvestLockLayer = g_thRowCropSystem:createInfoLayer("pfHarvestLockMap", pfSoilMapSize, 1, "modules/precisionFarming/maps", true)
                        if pfHarvestLockLayer == nil then
                            success = false
                        else
                            pfHarvestLockLayer.dmModifier = pfHarvestLockLayer:createModifier()
                            pfHarvestLockLayer.dmFilter = pfHarvestLockLayer:createFilter()
                            pfHarvestLockLayer.dmFilter:setValueCompareParams(DensityValueCompareType.EQUAL, 1)
                            self.infoLayers.harvestLock = pfHarvestLockLayer
                        end
                    end
                    if success then
                        self.infoLayers.isValid = true
                    end
                end
                if not self.infoLayers.isValid then
                    for _, pfInfoLayer in pairs(self.infoLayers) do
                        if type(pfInfoLayer) == THValueType.TABLE
                            and type(pfInfoLayer.delete) == THValueType.FUNCTION
                        then
                            pfInfoLayer:delete()
                        end
                    end
                end
                self.infoLayers.isLoadFinished = true
            end
        end)
        return ...
    end
    return appendFunc(superFunc(parent, mission, terrainNode, ...))
end
function THPFModule.overwrite_updateDestroyCommonArea(self, superFunc, sx, sz, wx, wz, hx, hz, ...)
    local function appendFunc(...)
        THUtils.pcall(function()
            if self.infoLayers.isValid then
                local thPFHarvestLockLayer = self.infoLayers.harvestLock
                thPFHarvestLockLayer:resetArea(sx, sz, wx, wz, hx, hz)
            end
        end)
        return ...
    end
    return appendFunc(superFunc(sx, sz, wx, wz, hx, hz, ...))
end
function THPFModule.overwrite_prepareFieldUpdateTask(self, superFunc, parent, ...)
    local function appendFunc(...)
        THUtils.pcall(function()
            if self.infoLayers.isValid then
                local thPFHarvestLockLayer = self.infoLayers.harvestLock
                local dmFieldPolygon = parent.area
                if dmFieldPolygon ~= nil then
                    thPFHarvestLockLayer:resetPolygon(dmFieldPolygon)
                end
            end
        end)
        return ...
    end
    return appendFunc(superFunc(parent, ...))
end
function THPFModule.overwrite_preUpdateCoverArea(self, superFunc, parent, fruitTypesArray, dmParallelogram, allowForageState, ...)
    THUtils.pcall(function()
        local mission = g_thRowCropSystem.coreData.mission
        local terrainNode = g_terrainNode
        if mission ~= nil and terrainNode ~= nil and self.infoLayers.isValid then
            local thAreaMaskLayer = g_thRowCropSystem.infoLayers.areaMask
            local dmAreaModifier = thAreaMaskLayer.dmModifier
            local thPFAreaMaskLayer = self.infoLayers.areaMask
            local dmPFAreaModifier = thPFAreaMaskLayer.dmModifier
            local thPFHarvestLockLayer = self.infoLayers.harvestLock
            local dmPFHarvestLockModifier = thPFHarvestLockLayer.dmModifier
            local hasRCSFruitType = false
            for _, fruitTypeIndex in pairs(fruitTypesArray) do
                local rcsFruitData = g_thRowCropSystem:getRowCropFruitType(fruitTypeIndex)
                if rcsFruitData ~= nil then
                    hasRCSFruitType = true
                end
            end
            if hasRCSFruitType then
                local dmFunctionData = THUtils.getFunctionCache(self, "CoverMap", "preUpdateCoverArea")
                if dmFunctionData == nil then
                    dmFunctionData = THUtils.createFunctionCache(self, "CoverMap", "preUpdateCoverArea", self.pfMod.environment)
                end
                dmFunctionData.superFunc = superFunc
                dmFunctionData.target = parent
                dmFunctionData.nitrogenMap = parent.pfModule.nitrogenMap
                dmFunctionData.pHMap = parent.pfModule.pHMap
                dmFunctionData.fruitTypesArray = fruitTypesArray
                dmFunctionData.fruitTypeIndex = FruitType.UNKNOWN
                dmFunctionData.maxChangedArea = 0
                dmFunctionData.lastHarvestLockArea = 0
                dmParallelogram:applyToModifier(dmAreaModifier)
                dmParallelogram:applyToModifier(dmPFAreaModifier)
                dmParallelogram:applyToModifier(dmPFHarvestLockModifier)
                g_thDensityMapController:setActiveFunctionData(dmFunctionData)
            end
        end
    end)
    return superFunc(parent, fruitTypesArray, dmParallelogram, allowForageState, ...)
end
function THPFModule.overwrite_postUpdateCoverArea(self, superFunc, parent, fruitTypesArray, dmParallelogram, allowForageState, ...)
    local function appendFunc(...)
        THUtils.pcall(function()
            local dmFunctionData = g_thDensityMapController:getActiveFunctionData()
            if dmFunctionData ~= nil and dmFunctionData.id == THPFModule.FUNCTION_CACHE_ID.UPDATE_COVER_AREA then
                if self.infoLayers.isValid then
                    local dmBaseUpdateModifiers = parent.densityMapModifiersUpdate
                    local thPFHarvestLockLayer = self.infoLayers.harvestLock
                    local dmPFHarvestLockModifier = thPFHarvestLockLayer.dmModifier
                    if dmBaseUpdateModifiers ~= nil then
                        local thFruitTypeData = g_thMapTypeManager:getFruitType(dmFunctionData.fruitTypeIndex)
                        if thFruitTypeData ~= nil and thFruitTypeData.index ~= FruitType.UNKNOWN then
                            if dmFunctionData.lastHarvestLockArea > 0 then
                                dmParallelogram:applyToModifier(dmPFHarvestLockModifier)
                                dmPFHarvestLockModifier:executeSet(1)
                            end
                        end
                    end
                end
                g_thDensityMapController:restoreLastActiveFunctionData()
            end
        end)
        return ...
    end
    return appendFunc(superFunc(parent, fruitTypesArray, dmParallelogram, allowForageState, ...))
end
THUtils.pcall(initScript)